//
//  To Do...
//
//    - Broadcast Rx
//      - Handle detune in twin note command
//      - Update property sheet (if visible) after command received
//
//    - Envelopes
//      - Allow a softy-like envelope to be configured
//      - Allow envelope to vary with note pitch - eg longer decay for low notes
//
//    - Filters
//      - Add a property sheet page for filter configuration
//      - Implement butterworth filters
//      - Add a filter configuration command
//      - Add filter parameters to program data
//
//    - Arpegiateur
//      - ?
//

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <windows.h>
#include <commctrl.h>

#include "../dsplib/dsplib.h"
#include "../dsplib/auxbus.h"
#include "../Ninereeds Broadcast Lib/Ninereeds Broadcast Lib.h"
#include "../nr_lib/nr_lib.h"

#include "resource.h"

#pragma optimize ("a", on)

#define MAX_TRACKS	8

float const oolog2 = 1.0 / log(2);

NR_LIB_NOTE_PAR		   (paraNoteInit,	"Note",			"Note"									)
NR_LIB_NOTE_PAR		   (paraNote,		"Bend Note",	"Bend Note"								)
NR_LIB_AMPLITUDE_PAR   (paraVolume,		"Volume",		"Volume (0=0%, 80=100%, FE=~200%)", 0x80)


CMachineParameter const *pParameters[] =
{
	// global

	// track
	&paraNoteInit,
	&paraNote,
	&paraVolume
};

#pragma pack(1)

//class gvals
//{
//public:
//	word bend;
//};

class tvals
{
public:
	byte noteinit;
	byte note;
	byte volume;
};

#pragma pack()

CMachineInfo const MacInfo =
{
	MT_GENERATOR, MI_VERSION, 0,				// type, version, flags
	1, MAX_TRACKS,								// min, max tracks
	0, 3, pParameters,							// num globalpars, num trackpars, *pars
	0,    NULL,									// num attribs, *attribs
	"Ninereeds NRS04", "NRS04", "Steve Horne",	// name, short name, author
	"Edit Ninereeds NRS04"						// command menu
};

class mi;

class CTrack
{
public:
	void Tick(tvals const &tv);
	void Stop();
	void Reset();
	bool Generate(float *psamples, int numsamples);

public:

	c_ADSR_Track   *f_Level;
	c_ADSR2_Track  *f_Mod_Level;
	c_Pitch_Track  *f_Pitch;
	c_Frq_To_Phase *f_Frq_To_Phase;
	c_Frq_To_Phase *f_Frq_To_Phase2;
	c_Waveform     *f_Waveform;  //  This just duplicates the pointer from mi::f_Waveform
	c_Fractal      *f_Fractal;   //  This just duplicates the pointer from mi::f_Fractal

	c_Frq_To_Phase *f_Mod_Frq_To_Phase;
	c_Waveform     *f_Mod_Waveform;
	c_Fractal      *f_Mod_Fractal;

	float f_Mod_Buffer [MAX_BUFFER_LENGTH];  //  Used for modulator

	float f_Buffer  [MAX_BUFFER_LENGTH];  //  Used for amplitude
	float f_Buffer2 [MAX_BUFFER_LENGTH];  //  Used for modulator envelope and twin note

	CTrack (c_ADSR_Global  *p_Level_Global,
          c_ADSR2_Global *p_Mod_Level_Global,
          c_Pitch_Global *p_Pitch_Global     )
	{
		f_Level            = new c_ADSR_Track  (*p_Level_Global);
		f_Mod_Level        = new c_ADSR2_Track (*p_Mod_Level_Global);
		f_Pitch            = new c_Pitch_Track (*p_Pitch_Global);
		f_Frq_To_Phase     = new c_Frq_To_Phase;
		f_Frq_To_Phase2    = new c_Frq_To_Phase;
    f_Mod_Frq_To_Phase = new c_Frq_To_Phase;

 		f_Level->Set_Volume (1.0);  //  Default note amplitude
	}

	~CTrack (void)
	{
		delete f_Level;
		delete f_Mod_Level;
		delete f_Pitch;
		delete f_Frq_To_Phase;
		delete f_Frq_To_Phase2;
	}

	mi *pmi;
};

#define MAX_PROGRAM_NAME 32
#define MAX_PROGRAMS     32
#define MAX_PROGRAM_IDS  32

struct c_Program
{
	char  f_Name [MAX_PROGRAM_NAME];

	byte  f_Waveform;
	byte  f_Depth;         float f_Effect_Hi;       float f_Effect_Lo;
	byte  f_Mod_Waveform;
	byte  f_Mod_Depth;         float f_Mod_Effect_Hi;       float f_Mod_Effect_Lo;
	float f_Attack;        float f_Decay;           float f_Sustain;           float f_Release;
	float f_Bend;
	bool  f_Enable_Twin;   bool  f_Fractal_Before;  float f_Detune_Semitones;  float f_Twin_Amp;
	int   f_Channel;
	bool  f_Ringmod_Main;  bool  f_Ringmod_Twin;
	bool  f_FM_Enable;     float f_FM_Level;   bool f_FM_Internal;
	bool  f_AM_Enable;     float f_AM_Level;   bool f_AM_Internal;
	bool  f_PWM_Enable;    float f_PWM_Level;  bool f_PWM_Internal;
  int   f_Mod_Frq_Type;  float f_Mod_Frq;

	float f_Mod_Attack;    float f_Mod_Decay;       float f_Mod_Sustain;       float f_Mod_Release;
	float f_Mod_Start;     float f_Mod_Peak;        float f_Mod_End;


	void Get_From_Current (mi *pmi);
	void Set_As_Current   (mi *pmi);

	static int Find (mi *pmi, char *p_Name);

	static void Fill_Program_Combo (mi *pmi, HWND hDlg);
	static void Fill_ID_Combo      (mi *pmi, HWND hDlg, int f_Curr_Program);
	static void Fill_ID_List       (mi *pmi, HWND hDlg);
};

struct c_V2_Program
{
	char  f_Name [MAX_PROGRAM_NAME];

	byte  f_Waveform;
	byte  f_Depth;         float f_Effect_Hi;       float f_Effect_Lo;
	byte  f_Mod_Waveform;
	byte  f_Mod_Depth;         float f_Mod_Effect_Hi;       float f_Mod_Effect_Lo;
	float f_Attack;        float f_Decay;           float f_Sustain;           float f_Release;
	float f_Bend;
	bool  f_Enable_Twin;   bool  f_Fractal_Before;  float f_Detune_Semitones;  float f_Twin_Amp;
	int   f_Channel;
	bool  f_Ringmod_Main;  bool  f_Ringmod_Twin;
	bool  f_FM_Enable;     float f_FM_Level;   bool f_FM_Internal;
	bool  f_AM_Enable;     float f_AM_Level;   bool f_AM_Internal;
	bool  f_PWM_Enable;    float f_PWM_Level;  bool f_PWM_Internal;
  int   f_Mod_Frq_Type;  float f_Mod_Frq;

  void Copy_To_Current (c_Program &p);
};

void c_V2_Program::Copy_To_Current (c_Program &p)
{
  memcpy (p.f_Name, f_Name, MAX_PROGRAM_NAME);

	p.f_Waveform = f_Waveform;
	p.f_Depth = f_Depth;
  p.f_Effect_Hi = f_Effect_Hi;
  p.f_Effect_Lo = f_Effect_Lo;
	p.f_Attack = f_Attack;
  p.f_Decay = f_Decay;
  p.f_Sustain = f_Sustain;
  p.f_Release = f_Release;
	p.f_Bend = f_Bend;
	p.f_Enable_Twin = f_Enable_Twin;
  p.f_Fractal_Before = f_Fractal_Before;
  p.f_Detune_Semitones = f_Detune_Semitones;
  p.f_Twin_Amp = f_Twin_Amp;
	p.f_Channel = f_Channel;
	p.f_Ringmod_Main = f_Ringmod_Main;
  p.f_Ringmod_Twin = f_Ringmod_Twin;
	p.f_FM_Enable = f_FM_Enable;
  p.f_FM_Level = f_FM_Level;
	p.f_AM_Enable = f_AM_Enable;
  p.f_AM_Level = f_AM_Level;
	p.f_PWM_Enable = f_PWM_Enable;
  p.f_PWM_Level = f_PWM_Level;

	p.f_Mod_Waveform = f_Mod_Waveform;
	p.f_Mod_Depth = f_Mod_Depth;
  p.f_Mod_Effect_Hi = f_Mod_Effect_Hi;
  p.f_Mod_Effect_Lo = f_Mod_Effect_Lo;
  p.f_FM_Internal = f_FM_Internal;
  p.f_AM_Internal = f_AM_Internal;
  p.f_PWM_Internal = f_PWM_Internal;
  p.f_Mod_Frq_Type = f_Mod_Frq_Type;
  p.f_Mod_Frq = f_Mod_Frq;

	p.f_Mod_Attack  = 0.2;  //  These defaults for back compatability rather than usefulness
  p.f_Mod_Decay   = 0.2;
  p.f_Mod_Sustain = 1.0;
  p.f_Mod_Release = 0.2;
	p.f_Mod_Start   = 1.0;
  p.f_Mod_Peak    = 1.0;
  p.f_Mod_End     = 1.0;
}

struct c_V1_Program
{
	char  f_Name [MAX_PROGRAM_NAME];

	byte  f_Waveform;
	byte  f_Depth;         float f_Effect_Hi;       float f_Effect_Lo;
	float f_Attack;        float f_Decay;           float f_Sustain;           float f_Release;
	float f_Bend;
	bool  f_Enable_Twin;   bool  f_Fractal_Before;  float f_Detune_Semitones;  float f_Twin_Amp;
	int   f_Channel;
	bool  f_Ringmod_Main;  bool  f_Ringmod_Twin;
	bool  f_FM_Enable;     float f_FM_Level;
	bool  f_AM_Enable;     float f_AM_Level;
	bool  f_PWM_Enable;    float f_PWM_Level;

  void Copy_To_Current (c_Program &p);
};

void c_V1_Program::Copy_To_Current (c_Program &p)
{
  memcpy (p.f_Name, f_Name, MAX_PROGRAM_NAME);

	p.f_Waveform = f_Waveform;
	p.f_Depth = f_Depth;
  p.f_Effect_Hi = f_Effect_Hi;
  p.f_Effect_Lo = f_Effect_Lo;
	p.f_Attack = f_Attack;
  p.f_Decay = f_Decay;
  p.f_Sustain = f_Sustain;
  p.f_Release = f_Release;
	p.f_Bend = f_Bend;
	p.f_Enable_Twin = f_Enable_Twin;
  p.f_Fractal_Before = f_Fractal_Before;
  p.f_Detune_Semitones = f_Detune_Semitones;
  p.f_Twin_Amp = f_Twin_Amp;
	p.f_Channel = f_Channel;
	p.f_Ringmod_Main = f_Ringmod_Main;
  p.f_Ringmod_Twin = f_Ringmod_Twin;
	p.f_FM_Enable = f_FM_Enable;
  p.f_FM_Level = f_FM_Level;
	p.f_AM_Enable = f_AM_Enable;
  p.f_AM_Level = f_AM_Level;
	p.f_PWM_Enable = f_PWM_Enable;
  p.f_PWM_Level = f_PWM_Level;

	p.f_Mod_Waveform = 0;
	p.f_Mod_Depth = 0;
  p.f_Mod_Effect_Hi = 0.0;
  p.f_Mod_Effect_Lo = 0.0;
  p.f_FM_Internal = false;
  p.f_AM_Internal = false;
  p.f_PWM_Internal = false;
  p.f_Mod_Frq_Type = 0;
  p.f_Mod_Frq = 0.0;

	p.f_Mod_Attack  = 0.2;
  p.f_Mod_Decay   = 0.2;
  p.f_Mod_Sustain = 1.0;
  p.f_Mod_Release = 0.2;
	p.f_Mod_Start   = 1.0;
  p.f_Mod_Peak    = 1.0;
  p.f_Mod_End     = 1.0;
}


//
//  Note multiple inheritence - easiest way to handle listener
//

class mi : public CMachineInterface, public c_Broadcast_Listener_CB
{
public:
	mi ();
	virtual ~mi();

	virtual void Init(CMachineDataInput  * const pi);
	virtual void Save(CMachineDataOutput * const po);

	virtual void Tick();
	virtual bool Work(float *psamples, int numsamples, int const mode);
	virtual void SetNumTracks(int const n);
	virtual void Stop();

  virtual void Command(int const i);

	virtual void Transpose_Notes  (int p_Semitones);

	virtual void All_Notes_Off    (void);
	virtual void All_Notes_Ignore (void);

  virtual void Set_Control           (int p_ID, int p_Value);


	virtual void Tick2 (void);
	virtual void Tick2 (int p_Channel, c_Broadcast_Params *p_Command);
	virtual bool Work2 (float *psamples, int numsamples, int const mode, int offset);

	void DisconnectAux ();

public:
	//  Do the following all need to be pointers?
	
	c_ADSR_Global  *f_Level;
	c_ADSR2_Global *f_Mod_Level;
	c_Pitch_Global *f_Pitch;
	c_Waveform     *f_Waveform;
	c_Fractal      *f_Fractal;
	c_Waveform     *f_Mod_Waveform;
	c_Fractal      *f_Mod_Fractal;

	//  Twin note parameters

	bool  f_Enable_Twin;
	bool  f_Fractal_Before;
	float f_Detune_Semitones;
	float f_Twin_Amp;

	int   f_Channel;  //  Auxbus Connection Channel

	bool  f_Listen_Enabled;  //  For Broadcast Commands
	int   f_Listen_Channel;

	bool  f_Ringmod_Main;
	bool  f_Ringmod_Twin;
	bool  f_FM_Enable;
	bool  f_AM_Enable;
	bool  f_PWM_Enable;
	bool  f_FM_Internal;
	bool  f_AM_Internal;
	bool  f_PWM_Internal;
	float f_FM_Level;
	float f_AM_Level;
	float f_PWM_Level;
  int   f_Mod_Frq_Type;
  float f_Mod_Frq;

	int       f_Num_Programs;
	c_Program f_Programs     [MAX_PROGRAMS];
	int       f_Program_IDs  [MAX_PROGRAM_IDS];  //  Max program ID to f_Programs index

	float  f_Aux_Buf_Full [MAX_BUFFER_LENGTH];  //  Used for AuxBus input
	float *f_Aux_Buffer;

	int numTracks;
	CTrack *Tracks[MAX_TRACKS];

	tvals tval     [MAX_TRACKS];
//	gvals gval;

};

void c_Program::Get_From_Current (mi *pmi)
{
	f_Waveform  = pmi->f_Waveform->Get_Waveform ();

	f_Depth     = pmi->f_Fractal->Get_Depth     ();
	f_Effect_Hi = pmi->f_Fractal->Get_Effect_Hi ();
	f_Effect_Lo = pmi->f_Fractal->Get_Effect_Lo ();

	f_Mod_Waveform  = pmi->f_Mod_Waveform->Get_Waveform ();

	f_Mod_Depth     = pmi->f_Mod_Fractal->Get_Depth     ();
	f_Mod_Effect_Hi = pmi->f_Mod_Fractal->Get_Effect_Hi ();
	f_Mod_Effect_Lo = pmi->f_Mod_Fractal->Get_Effect_Lo ();

  f_Mod_Frq_Type  = pmi->f_Mod_Frq_Type;
  f_Mod_Frq       = pmi->f_Mod_Frq;

	f_Attack    = pmi->f_Level->Get_Attack  ();
	f_Decay     = pmi->f_Level->Get_Decay   ();
	f_Sustain   = pmi->f_Level->Get_Sustain ();
	f_Release   = pmi->f_Level->Get_Release ();

	f_Mod_Attack    = pmi->f_Mod_Level->Get_Attack  ();
	f_Mod_Decay     = pmi->f_Mod_Level->Get_Decay   ();
	f_Mod_Sustain   = pmi->f_Mod_Level->Get_Sustain ();
	f_Mod_Release   = pmi->f_Mod_Level->Get_Release ();
	f_Mod_Start     = pmi->f_Mod_Level->Get_Start   ();
	f_Mod_Peak      = pmi->f_Mod_Level->Get_Peak    ();
	f_Mod_End       = pmi->f_Mod_Level->Get_End     ();

	f_Bend      = pmi->f_Pitch->Get_Attack  ();

	f_Enable_Twin      = pmi->f_Enable_Twin;
	f_Fractal_Before   = pmi->f_Fractal_Before;
	f_Detune_Semitones = pmi->f_Detune_Semitones;
	f_Twin_Amp         = pmi->f_Twin_Amp;

	f_Channel          = pmi->f_Channel;

	f_Ringmod_Main     = pmi->f_Ringmod_Main;
	f_Ringmod_Twin     = pmi->f_Ringmod_Twin;

	f_FM_Enable        = pmi->f_FM_Enable;
	f_FM_Internal      = pmi->f_FM_Internal;
	f_FM_Level         = pmi->f_FM_Level;

	f_AM_Enable        = pmi->f_AM_Enable;
	f_AM_Internal      = pmi->f_AM_Internal;
	f_AM_Level         = pmi->f_AM_Level;

	f_PWM_Enable       = pmi->f_PWM_Enable;
	f_PWM_Internal     = pmi->f_PWM_Internal;
	f_PWM_Level        = pmi->f_PWM_Level;
}

void c_Program::Set_As_Current   (mi *pmi)
{
	pmi->f_Waveform->Set_Waveform (f_Waveform);

	pmi->f_Fractal->Set_Depth     (f_Depth);
	pmi->f_Fractal->Set_Effect_Hi (f_Effect_Hi);
	pmi->f_Fractal->Set_Effect_Lo (f_Effect_Lo);

	pmi->f_Mod_Waveform->Set_Waveform (f_Mod_Waveform);

	pmi->f_Mod_Fractal->Set_Depth     (f_Mod_Depth);
	pmi->f_Mod_Fractal->Set_Effect_Hi (f_Mod_Effect_Hi);
	pmi->f_Mod_Fractal->Set_Effect_Lo (f_Mod_Effect_Lo);

  pmi->f_Mod_Frq_Type  = f_Mod_Frq_Type;
  pmi->f_Mod_Frq       = f_Mod_Frq;

	pmi->f_Level->Set_Attack  (f_Attack);
	pmi->f_Level->Set_Decay   (f_Decay);
	pmi->f_Level->Set_Sustain (f_Sustain);
	pmi->f_Level->Set_Release (f_Release);

	pmi->f_Mod_Level->Set_Attack  (f_Mod_Attack);
	pmi->f_Mod_Level->Set_Decay   (f_Mod_Decay);
	pmi->f_Mod_Level->Set_Sustain (f_Mod_Sustain);
	pmi->f_Mod_Level->Set_Release (f_Mod_Release);
	pmi->f_Mod_Level->Set_Start   (f_Mod_Start);
	pmi->f_Mod_Level->Set_Peak    (f_Mod_Peak);
	pmi->f_Mod_Level->Set_End     (f_Mod_End);

	pmi->f_Pitch->Set_Attack  (f_Bend);

	pmi->f_Enable_Twin      = f_Enable_Twin;
	pmi->f_Fractal_Before   = f_Fractal_Before;
	pmi->f_Detune_Semitones = f_Detune_Semitones;
	pmi->f_Twin_Amp         = f_Twin_Amp;

	pmi->f_Channel          = f_Channel;

	pmi->f_Ringmod_Main     = f_Ringmod_Main;
	pmi->f_Ringmod_Twin     = f_Ringmod_Twin;

	pmi->f_FM_Enable        = f_FM_Enable;
	pmi->f_FM_Internal      = f_FM_Internal;
	pmi->f_FM_Level         = f_FM_Level;

	pmi->f_AM_Enable        = f_AM_Enable;
	pmi->f_AM_Internal      = f_AM_Internal;
	pmi->f_AM_Level         = f_AM_Level;

	pmi->f_PWM_Enable       = f_PWM_Enable;
	pmi->f_PWM_Internal     = f_PWM_Internal;
	pmi->f_PWM_Level        = f_PWM_Level;
}

int c_Program::Find (mi *pmi, char *p_Name)
{
	for (int i = 0; i < pmi->f_Num_Programs; i++)
	{
		if (strcmp (p_Name, pmi->f_Programs [i].f_Name) == 0)
		{
			return i;
		}
	}

	return -1;
}

void c_Program::Fill_Program_Combo (mi *pmi, HWND hDlg)
{
	SendMessage (GetDlgItem (hDlg, IDC_COMBO_PROG_NAME), CB_RESETCONTENT, 0, 0L);

	for (int i = 0; i < pmi->f_Num_Programs; i++)
	{
		SendMessage (GetDlgItem (hDlg, IDC_COMBO_PROG_NAME),
					 CB_ADDSTRING, 0, (LONG) (LPSTR) pmi->f_Programs[i].f_Name);
	}
}

void c_Program::Fill_ID_Combo (mi *pmi, HWND hDlg, int f_Curr_Program)
{
	char l_Buf [80];

	SendMessage (GetDlgItem (hDlg, IDC_COMBO_PRG_IDS), CB_RESETCONTENT, 0, 0L);

	for (int i = 0; i < MAX_PROGRAM_IDS; i++)
	{
		if (pmi->f_Program_IDs [i] == f_Curr_Program)
		{
			sprintf (l_Buf, "%d", i);

			SendMessage (GetDlgItem (hDlg, IDC_COMBO_PRG_IDS),
						 CB_ADDSTRING, 0, (LONG) (LPSTR) l_Buf);
		}
	}
}

void c_Program::Fill_ID_List (mi *pmi, HWND hDlg)
{
	char l_Buf [80];

	ListView_DeleteAllItems (GetDlgItem (hDlg, IDC_LIST_PROG_IDS));

	LV_ITEM l_Item;

	l_Item.mask       = LVIF_TEXT | LVIF_PARAM;
	l_Item.iItem      = 0;
	l_Item.iSubItem   = 0;
	l_Item.state      = 0;
	l_Item.stateMask  = 0;
	l_Item.pszText    = NULL;
	l_Item.cchTextMax = 32;
	l_Item.iImage     = 0;
	l_Item.lParam     = 0;

	for (int i = 0; i < MAX_PROGRAM_IDS; i++)
	{
		sprintf (l_Buf, "%d", i);

		l_Item.iItem    = i;
		l_Item.lParam   = i;
		l_Item.pszText  = l_Buf;
		l_Item.iSubItem = 0;

		ListView_InsertItem (GetDlgItem (hDlg, IDC_LIST_PROG_IDS), &l_Item);

		l_Item.iItem    = i;
		l_Item.lParam   = i;
		l_Item.pszText  = pmi->f_Programs [pmi->f_Program_IDs [i]].f_Name;
		l_Item.iSubItem = 1;

		ListView_InsertItem (GetDlgItem (hDlg, IDC_LIST_PROG_IDS), &l_Item);
	}
}


DLL_EXPORTS
NR_STD_GENERATOR_STOP

void mi::SetNumTracks(int const n)
{
	for (int i = numTracks; i < n; i++)
	{
		Tracks[i]->Reset ();
	}

	numTracks = n;
}


//  Note that although Work itself exists mainly to redirect things to the
//  broadcast library stuff, it can do some things itself - in this case,
//  it retrieves the AuxBus input (if needed).
//
//  Work2, however, does the real signal generation.

bool mi::Work (float *psamples, int numsamples, int const mode)
{
	if (((mode & WM_WRITE) != 0) && (f_Channel != -1))
	{
		AB_Receive (f_Channel, f_Aux_Buf_Full, numsamples);
	}

	return Handle_Work (psamples, numsamples, mode);
}


bool mi::Work2(float *psamples, int numsamples, int const mode, int offset)
{
//	if ((mode & WM_WRITE) == 0)
//	{
//		return false;
//	}

	bool gotsomething = false;
	bool l_Temp;

	f_Aux_Buffer = f_Aux_Buf_Full + offset;

	float *paux = pCB->GetAuxBuffer();

	for (int c = 0; c < numTracks; c++)
	{
		if (gotsomething)
		{
			l_Temp = Tracks[c]->Generate(paux, numsamples);

			gotsomething |= l_Temp;

			if (l_Temp)
			{
				DSP_Add(psamples, paux, numsamples);
			}
		}
		else
		{
			gotsomething |= Tracks[c]->Generate(psamples, numsamples);
		}
	}

	return gotsomething;
}

mi::mi()
{
	f_Level    = NULL;
	f_Mod_Level = NULL;
	f_Pitch    = NULL;
	f_Waveform = NULL;
	f_Fractal  = NULL;
	f_Mod_Waveform = NULL;
	f_Mod_Fractal  = NULL;

	f_Enable_Twin      = true;
	f_Fractal_Before   = true;
	f_Detune_Semitones = 0.1;
	f_Twin_Amp         = 1.0;

	f_Channel = -1;

	f_Listen_Enabled = false;
	f_Listen_Channel = 0;

	f_Ringmod_Main = false;
	f_Ringmod_Twin = false;
	f_FM_Enable    = false;
	f_FM_Internal  = true;
	f_FM_Level     = 0.0;
	f_AM_Enable    = false;
	f_AM_Internal  = true;
	f_AM_Level     = 0.0;
	f_PWM_Enable   = false;
	f_PWM_Internal = true;
	f_PWM_Level    = 0.0;

  f_Mod_Frq_Type = 0;
  f_Mod_Frq      = 0.0;

	f_Num_Programs = 0;

	for (int i = 0; i < MAX_TRACKS; i++)
	{
		Tracks [i] = NULL;
	}

	for (i = 0; i < MAX_PROGRAM_IDS; i++)
	{
		f_Program_IDs [i] = -1;
	}

//	GlobalVals = &gval;
	GlobalVals = NULL;
	TrackVals  = tval;
	AttrVals   = NULL;
}

mi::~mi()
{
	AB_Disconnect (this);

	if (f_Listen_Enabled)
	{
		NR_Stop_Listening (f_Listen_Channel, this);
	}

	for (int i = 0; i < MAX_TRACKS; i++)
	{
		delete Tracks [i];
	}

	delete f_Waveform;
	delete f_Pitch;
	delete f_Level;
	delete f_Mod_Level;
	delete f_Fractal;
	delete f_Mod_Waveform;
	delete f_Mod_Fractal;
}

void mi::DisconnectAux()
{
	MACHINE_LOCK;

	f_Channel = -1;
}

void cb(void *user)  //  Auxbus disconnection callback
{
	mi *pmi = (mi *)user;
	pmi->DisconnectAux();
}

void mi::Init(CMachineDataInput * const pi)
{
	f_Level     = new c_ADSR_Global;
	f_Mod_Level = new c_ADSR2_Global;
  f_Pitch     = new c_Pitch_Global (pMasterInfo->SamplesPerSec);
	f_Waveform  = new c_Waveform;
	f_Fractal   = new c_Fractal;
	f_Mod_Waveform = new c_Waveform;
	f_Mod_Fractal  = new c_Fractal;

	f_Level->Set_Samples_Per_Sec     (pMasterInfo->SamplesPerSec);
	f_Mod_Level->Set_Samples_Per_Sec (pMasterInfo->SamplesPerSec);

  for (int i = 0; i < MAX_TRACKS; i++)
	{
		Tracks [i] = new CTrack (f_Level, f_Mod_Level, f_Pitch);

		Tracks [i]->pmi = this;
		Tracks [i]->f_Waveform     = f_Waveform;
		Tracks [i]->f_Fractal      = f_Fractal;
		Tracks [i]->f_Mod_Waveform = f_Mod_Waveform;
		Tracks [i]->f_Mod_Fractal  = f_Mod_Fractal;

		Tracks [i]->Reset();
	}

	if (pi != NULL)
	{
		word l_Version;
		pi->Read (l_Version);

		while (l_Version != 0x0000)
		{
			switch (l_Version)
			{
				case 0x0101 :
				{
					byte  l_Waveform;   pi->Read (l_Waveform );  f_Waveform->Set_Waveform (l_Waveform);

					f_Waveform->Set_Optimise_Enable  (false);
					f_Waveform->Set_Optimise_Quality (NR_LIB_WAVEFORM_MAX_QUALITY);

					break;
				}
				case 0x0102 :
				{
					byte  l_Waveform;   pi->Read (l_Waveform );  f_Waveform->Set_Waveform         (l_Waveform);
					bool  l_Optimise;   pi->Read (l_Optimise );  f_Waveform->Set_Optimise_Enable  (l_Optimise);
					int   l_Quality;    pi->Read (l_Quality  );  f_Waveform->Set_Optimise_Quality (l_Quality );

          break;
        }
				case 0x0201 :
				{
					byte  l_Depth;      pi->Read (l_Depth    );  f_Fractal->Set_Depth     (l_Depth);
					float l_Effect_Hi;  pi->Read (l_Effect_Hi);  f_Fractal->Set_Effect_Hi (l_Effect_Hi);
					float l_Effect_Lo;  pi->Read (l_Effect_Lo);  f_Fractal->Set_Effect_Lo (l_Effect_Lo);

					f_Fractal->Set_Optimise_Enable  (false);
					f_Fractal->Set_Optimise_Quality (NR_LIB_FRACTAL_MAX_QUALITY);

					break;
				}
				case 0x0202 :
				{
					byte  l_Depth;      pi->Read (l_Depth    );  f_Fractal->Set_Depth     (l_Depth);
					float l_Effect_Hi;  pi->Read (l_Effect_Hi);  f_Fractal->Set_Effect_Hi (l_Effect_Hi);
					float l_Effect_Lo;  pi->Read (l_Effect_Lo);  f_Fractal->Set_Effect_Lo (l_Effect_Lo);
					bool  l_Optimise;   pi->Read (l_Optimise );  f_Fractal->Set_Optimise_Enable  (l_Optimise);
					int   l_Quality;    pi->Read (l_Quality  );  f_Fractal->Set_Optimise_Quality (l_Quality );

					break;
				}
				case 0x0301 :
				{
					float l_Attack;     pi->Read (l_Attack   );  f_Level->Set_Attack      (l_Attack);
					float l_Decay;      pi->Read (l_Decay    );  f_Level->Set_Decay       (l_Decay);
					float l_Sustain;    pi->Read (l_Sustain  );  f_Level->Set_Sustain     (l_Sustain);
					float l_Release;    pi->Read (l_Release  );  f_Level->Set_Release     (l_Release);

					break;
				}
				case 0x0401 :
				{
					float l_Bend;       pi->Read (l_Bend     );  f_Pitch->Set_Attack      (l_Bend);

					break;
				}
				case 0x0501 :
				{
					pi->Read (f_Enable_Twin     );
					pi->Read (f_Fractal_Before  );
					pi->Read (f_Detune_Semitones);
					f_Twin_Amp = 1.0;

					break;
				}
				case 0x0502 :
				{
					pi->Read (f_Enable_Twin     );
					pi->Read (f_Fractal_Before  );
					pi->Read (f_Detune_Semitones);
					pi->Read (f_Twin_Amp        );

					break;
				}
				case 0x0601 :
				{
					pi->Read (f_Channel         );
					pi->Read (f_Ringmod_Main	);
					pi->Read (f_Ringmod_Twin	);
					pi->Read (f_FM_Level        );

					f_FM_Enable = (f_FM_Level > 0.0);

					f_AM_Enable = false;
					f_AM_Level  = 0.25;

					f_PWM_Enable = false;
					f_PWM_Level  = 0.25;

          f_FM_Internal  = false;
          f_AM_Internal  = false;
          f_PWM_Internal = false;

					break;
				}
				case 0x0602 :
				{
					pi->Read (f_Channel         );
					pi->Read (f_Ringmod_Main	);
					pi->Read (f_Ringmod_Twin	);
					pi->Read (f_FM_Enable       );
					pi->Read (f_FM_Level        );

					f_AM_Enable = false;
					f_AM_Level  = 0.25;

					f_PWM_Enable = false;
					f_PWM_Level  = 0.25;

          f_FM_Internal  = false;
          f_AM_Internal  = false;
          f_PWM_Internal = false;

					break;
				}
				case 0x0603 :
				{
					pi->Read (f_Channel         );
					pi->Read (f_Ringmod_Main	);
					pi->Read (f_Ringmod_Twin	);
					pi->Read (f_FM_Enable       );
					pi->Read (f_FM_Level        );
					pi->Read (f_AM_Enable       );
					pi->Read (f_AM_Level        );

					f_PWM_Enable = false;
					f_PWM_Level  = 0.25;
          f_FM_Internal  = false;
          f_AM_Internal  = false;
          f_PWM_Internal = false;

					break;
				}
				case 0x0604 :
				{
					pi->Read (f_Channel         );
					pi->Read (f_Ringmod_Main	);
					pi->Read (f_Ringmod_Twin	);
					pi->Read (f_FM_Enable       );
					pi->Read (f_FM_Level        );
					pi->Read (f_AM_Enable       );
					pi->Read (f_AM_Level        );
					pi->Read (f_PWM_Enable      );
					pi->Read (f_PWM_Level       );

          f_FM_Internal  = false;
          f_AM_Internal  = false;
          f_PWM_Internal = false;

					break;
				}
				case 0x0605 :
				{
					pi->Read (f_Channel         );
					pi->Read (f_Ringmod_Main	);
					pi->Read (f_Ringmod_Twin	);
					pi->Read (f_FM_Enable       );
					pi->Read (f_FM_Internal     );
					pi->Read (f_FM_Level        );
					pi->Read (f_AM_Enable       );
					pi->Read (f_AM_Internal     );
					pi->Read (f_AM_Level        );
					pi->Read (f_PWM_Enable      );
					pi->Read (f_PWM_Internal    );
					pi->Read (f_PWM_Level       );

					break;
				}
				case 0x0701 :
				{
					pi->Read (f_Listen_Enabled  );
					pi->Read (f_Listen_Channel	);

					break;
				}
				case 0x0801 :
				{
          c_V1_Program l_Temp;

					pi->Read (f_Num_Programs);

					for (int i = 0; i < f_Num_Programs; i++)
					{
						pi->Read (&l_Temp, sizeof (c_V1_Program));

            l_Temp.Copy_To_Current (f_Programs [i]);
					}

					break;
				}
				case 0x0802 :
				{
          c_V2_Program l_Temp;

					pi->Read (f_Num_Programs);

					for (int i = 0; i < f_Num_Programs; i++)
					{
						pi->Read (&l_Temp, sizeof (c_V2_Program));

            l_Temp.Copy_To_Current (f_Programs [i]);
					}

					break;
				}
				case 0x0803 :
				{
					pi->Read (f_Num_Programs);

					for (int i = 0; i < f_Num_Programs; i++)
					{
						pi->Read (&f_Programs [i], sizeof (c_Program));
					}

					break;
				}
				case 0x0901 :
				{
					for (int i = 0; i < MAX_PROGRAM_IDS; i++)
					{
						pi->Read (f_Program_IDs [i]);
					}

					break;
				}
				case 0x0A01 :
				{
          Read_Controls (pi);

          break;
        }
				case 0x0B01 :
				{
					byte  l_Waveform;   pi->Read (l_Waveform );  f_Mod_Waveform->Set_Waveform         (l_Waveform);
					bool  l_Optimise;   pi->Read (l_Optimise );  f_Mod_Waveform->Set_Optimise_Enable  (l_Optimise);
					int   l_Quality;    pi->Read (l_Quality  );  f_Mod_Waveform->Set_Optimise_Quality (l_Quality );

          break;
        }
				case 0x0C01 :
				{
					byte  l_Depth;      pi->Read (l_Depth    );  f_Mod_Fractal->Set_Depth     (l_Depth);
					float l_Effect_Hi;  pi->Read (l_Effect_Hi);  f_Mod_Fractal->Set_Effect_Hi (l_Effect_Hi);
					float l_Effect_Lo;  pi->Read (l_Effect_Lo);  f_Mod_Fractal->Set_Effect_Lo (l_Effect_Lo);
					bool  l_Optimise;   pi->Read (l_Optimise );  f_Mod_Fractal->Set_Optimise_Enable  (l_Optimise);
					int   l_Quality;    pi->Read (l_Quality  );  f_Mod_Fractal->Set_Optimise_Quality (l_Quality );

					break;
				}
				case 0x0D01 :
				{
					pi->Read (f_Mod_Frq_Type);
					pi->Read (f_Mod_Frq     );

					break;
				}
				case 0x0E01 :
				{
					float l_Attack;     pi->Read (l_Attack   );  f_Mod_Level->Set_Attack      (l_Attack);
					float l_Decay;      pi->Read (l_Decay    );  f_Mod_Level->Set_Decay       (l_Decay);
					float l_Sustain;    pi->Read (l_Sustain  );  f_Mod_Level->Set_Sustain     (l_Sustain);
					float l_Release;    pi->Read (l_Release  );  f_Mod_Level->Set_Release     (l_Release);
					float l_Start;      pi->Read (l_Start    );  f_Mod_Level->Set_Start       (l_Start);
					float l_Peak;       pi->Read (l_Peak     );  f_Mod_Level->Set_Peak        (l_Peak);
					float l_End;        pi->Read (l_End      );  f_Mod_Level->Set_End         (l_End);

					break;
				}
			}

			pi->Read (l_Version);
		}
	}

	Set_Master_Info (pMasterInfo);

	if (f_Channel != -1)
	{
		AB_ConnectOutput (f_Channel, MacInfo.ShortName, cb, this);
	}

	if (f_Listen_Enabled)
	{
		NR_Start_Listening (f_Listen_Channel, this);  //  Note - ignoring possibility of error
	}
}

void mi::Save(CMachineDataOutput * const po)
{
	//  Each functional block is identified by a code which indicates the type of data and
	//  the version number, thus allowing extensions to be added with the minimum of hassle.
	//
	//  A size value for each block may be a good idea, to allow unknown blocks to be skipped.
	//  This will ensure forward compatability.

	if (po != NULL)
	{
		po->Write ((word) 0x0102);							//  Waveform version 2
		po->Write ((byte) f_Waveform->Get_Waveform         ());
    po->Write (       f_Waveform->Get_Optimise_Enable  ());
    po->Write (       f_Waveform->Get_Optimise_Quality ());

		po->Write ((word) 0x0202);							//  Fractal version 2
		po->Write ((byte) f_Fractal->Get_Depth     ());
		po->Write (       f_Fractal->Get_Effect_Hi ());
		po->Write (       f_Fractal->Get_Effect_Lo ());
    po->Write (       f_Fractal->Get_Optimise_Enable  ());
    po->Write (       f_Fractal->Get_Optimise_Quality ());

		po->Write ((word) 0x0301);							//  Envelope version 1
		po->Write (       f_Level->Get_Attack  ());
		po->Write (       f_Level->Get_Decay   ());
		po->Write (       f_Level->Get_Sustain ());
		po->Write (       f_Level->Get_Release ());

		po->Write ((word) 0x0401);							//  Pitch bend version 1
		po->Write (       f_Pitch->Get_Attack  ());

		po->Write ((word) 0x0502);							//  Twin note version 1
		po->Write (       f_Enable_Twin     );
		po->Write (       f_Fractal_Before  );
		po->Write (       f_Detune_Semitones);
		po->Write (       f_Twin_Amp        );

		po->Write ((word) 0x0605);							//  AuxBus Effects
		po->Write (       f_Channel         );
		po->Write (       f_Ringmod_Main    );
		po->Write (       f_Ringmod_Twin    );
		po->Write (       f_FM_Enable       );
		po->Write (       f_FM_Internal     );
		po->Write (       f_FM_Level        );
		po->Write (       f_AM_Enable       );
		po->Write (       f_AM_Internal     );
		po->Write (       f_AM_Level        );
		po->Write (       f_PWM_Enable      );
		po->Write (       f_PWM_Internal    );
		po->Write (       f_PWM_Level       );

		po->Write ((word) 0x0701);							//  Broadcast Rx
		po->Write (       f_Listen_Enabled  );
		po->Write (       f_Listen_Channel  );

		po->Write ((word) 0x0803);							//  Program Bank
		po->Write (       f_Num_Programs    );

		for (int i = 0; i < f_Num_Programs; i++)
		{
			po->Write (&f_Programs [i], sizeof (c_Program));
		}

		po->Write ((word) 0x0901);							//  Program ID Mappings

		for (i = 0; i < MAX_PROGRAM_IDS; i++)
		{
			po->Write (f_Program_IDs [i]);
		}

		po->Write ((word) 0x0A01);							//  Control Mappings
    Write_Controls (po);

		po->Write ((word) 0x0B01);							//  Modulator Waveform version 1
		po->Write ((byte) f_Mod_Waveform->Get_Waveform         ());
    po->Write (       f_Mod_Waveform->Get_Optimise_Enable  ());
    po->Write (       f_Mod_Waveform->Get_Optimise_Quality ());

		po->Write ((word) 0x0C01);							//  Modulator Fractal version 1
		po->Write ((byte) f_Mod_Fractal->Get_Depth     ());
		po->Write (       f_Mod_Fractal->Get_Effect_Hi ());
		po->Write (       f_Mod_Fractal->Get_Effect_Lo ());
    po->Write (       f_Mod_Fractal->Get_Optimise_Enable  ());
    po->Write (       f_Mod_Fractal->Get_Optimise_Quality ());

		po->Write ((word) 0x0D01);							//  Modulator Frq Settings
		po->Write (       f_Mod_Frq_Type);
		po->Write (       f_Mod_Frq     );

		po->Write ((word) 0x0E01);							//  Modulator Envelope version 1
		po->Write (       f_Mod_Level->Get_Attack  ());
		po->Write (       f_Mod_Level->Get_Decay   ());
		po->Write (       f_Mod_Level->Get_Sustain ());
		po->Write (       f_Mod_Level->Get_Release ());
		po->Write (       f_Mod_Level->Get_Start   ());
		po->Write (       f_Mod_Level->Get_Peak    ());
		po->Write (       f_Mod_Level->Get_End     ());

		po->Write ((word) 0x0000);							//  Terminator
	}
}

void mi::Tick()
{
	Handle_Tick ();
}

void mi::Tick2 (void)
{
	for (int c = 0; c < numTracks; c++)
	{
		Tracks[c]->Tick(tval[c]);
	}
}

void mi::Transpose_Notes  (int p_Semitones)
{
  int l_Note;

  for (int i = 0; i < MAX_TRACKS; i++)
  {
    if (   (tval [i].noteinit != NOTE_NO )
        && (tval [i].noteinit != NOTE_OFF))
    {
      l_Note  = ((tval [i].noteinit >> 4) * 12) + (tval [i].noteinit & 0x0F);
      l_Note += p_Semitones;
      
      if ((l_Note >= NOTE_MIN) && (l_Note <= NOTE_MAX))
      {
        tval [i].noteinit = ((l_Note / 12) << 4) + (l_Note % 12);
      }
      else
      {
        tval [i].noteinit = NOTE_OFF;
      }
    }

    if (   (tval [i].note != NOTE_NO )
        && (tval [i].note != NOTE_OFF))
    {
      l_Note  = ((tval [i].note >> 4) * 12) + (tval [i].note & 0x0F);
      l_Note += p_Semitones;
      
      if ((l_Note >= NOTE_MIN) && (l_Note <= NOTE_MAX))
      {
        tval [i].note = ((l_Note / 12) << 4) + (l_Note % 12);
      }
      else
      {
        tval [i].noteinit = NOTE_OFF;
        tval [i].note     = NOTE_NO;
      }
    }
  }
}

void mi::All_Notes_Off    (void)
{
  for (int i = 0; i < MAX_TRACKS; i++)
  {
    tval [i].noteinit = NOTE_OFF;
  }
}

void mi::All_Notes_Ignore (void)
{
  for (int i = 0; i < MAX_TRACKS; i++)
  {
    if (   (tval [i].noteinit != NOTE_NO )
        && (tval [i].noteinit != NOTE_OFF))
    {
      tval [i].noteinit = NOTE_NO;
    }

    tval [i].note = NOTE_NO;
  }
}

void mi::Set_Control (int p_ID, int p_Value)
{
  switch (p_ID)
  {
    case 0 :  //  Volume
    {
      int k = p_Value >> 8;

      if (k > 0xFE)  {  k = 0xFE;  }

      for (int i = 0; i < MAX_TRACKS; i++)
      {
        tval [i].volume = k;
      }

      break;
    }
    case 1 :  //  Attack
    {
      float l_Temp = 10000.0 * (((float) p_Value) / 65534.0);
      f_Level->Set_Attack (l_Temp);

      break;
    }
    case 2 :  //  Decay
    {
      float l_Temp = 10000.0 * (((float) p_Value) / 65534.0);
      f_Level->Set_Decay (l_Temp);

      break;
    }
    case 3 :  //  Sustain
    {
      float l_Temp = (((float) p_Value) / 65534.0);
      f_Level->Set_Sustain (l_Temp);

      break;
    }
    case 4 :  //  Release
    {
      float l_Temp = 10000.0 * (((float) p_Value) / 65534.0);
      f_Level->Set_Release (l_Temp);

      break;
    }
    case 5 :  //  Bend Rate
    {
      float l_Temp = 10000.0 * (((float) p_Value) / 65534.0);
      f_Pitch->Set_Attack (l_Temp);

      break;
    }
    case 6 :  //  Fractal Effect Low
    {
      float l_Effect = 9.0 * (((float) p_Value) / 65534.0);
      f_Fractal->Set_Effect_Lo (l_Effect);

      break;
    }
    case 7 :  //  Fractal Effect High
    {
      float l_Effect = 9.0 * (((float) p_Value) / 65534.0);
      f_Fractal->Set_Effect_Hi (l_Effect);

      break;
    }
    case 8 :  //  Modulator Fractal Effect Low
    {
      float l_Effect = 9.0 * (((float) p_Value) / 65534.0);
      f_Fractal->Set_Effect_Lo (l_Effect);

      break;
    }
    case 9 :  //  Modulator Fractal Effect High
    {
      float l_Effect = 9.0 * (((float) p_Value) / 65534.0);
      f_Fractal->Set_Effect_Hi (l_Effect);

      break;
    }
  }
}

void mi::Tick2 (int p_Channel, c_Broadcast_Params *p_Command)
{
	switch (p_Command->p_Command)
	{
	//  Commands 0x01 to ? will be standardised (may have 0x40-0x7F as macros of some form?)
	case 0x01 :  //  Program Select
		{
			if (   (p_Command->p_Byte_Param1                 >= 0              )
				&& (p_Command->p_Byte_Param1                 <  MAX_PROGRAM_IDS)
				&& (f_Program_IDs [p_Command->p_Byte_Param1] != -1             ))
			{
				f_Programs [f_Program_IDs [p_Command->p_Byte_Param1]].Set_As_Current (this);
			}

			break;
		}

	//  Setting a convention where commands 0x80 upwards are usable for specific purposes
	//  by each machine, and need not be user configurable, will allow easy setup of basic
	//  commands. They will, however, need to be documented for each machine - and there
	//  will be a need to select a channel to listen to.
	case 0x80 :
		{
			//  Set waveform from p_Byte_Param1

			if (   (p_Command->p_Byte_Param1 >= 1                   )
				&& (p_Command->p_Byte_Param1 <= NR_LIB_HIGH_WAVEFORM))
			{
				f_Waveform->Set_Waveform (p_Command->p_Byte_Param1);
			}

			break;
		}
	case 0x81 :
		{
			//  Set fractal

			if (   (p_Command->p_Byte_Param1 >=  0)
				&& (p_Command->p_Byte_Param1 <= 10))
			{
				f_Fractal->Set_Depth (p_Command->p_Byte_Param1);
			}

			if (p_Command->p_Word_Param1 != 0xFFFF)
			{
				float l_Effect = 9.0 * (((float) p_Command->p_Word_Param1) / 65534.0);
				f_Fractal->Set_Effect_Lo (l_Effect);
			}

			if (p_Command->p_Word_Param2 != 0xFFFF)
			{
				float l_Effect = 9.0 * (((float) p_Command->p_Word_Param2) / 65534.0);
				f_Fractal->Set_Effect_Hi (l_Effect);
			}

			break;
		}
	case 0x82 :
		{
			//  Set ADSR

			if (p_Command->p_Word_Param1 != 0xFFFF)
			{
				float l_Temp = 10000.0 * (((float) p_Command->p_Word_Param1) / 65534.0);
				f_Level->Set_Attack (l_Temp);
			}

			if (p_Command->p_Word_Param2 != 0xFFFF)
			{
				float l_Temp = 10000.0 * (((float) p_Command->p_Word_Param2) / 65534.0);
				f_Level->Set_Decay (l_Temp);
			}

			if (p_Command->p_Word_Param3 != 0xFFFF)
			{
				float l_Temp = (((float) p_Command->p_Word_Param3) / 65534.0);
				f_Level->Set_Sustain (l_Temp);
			}

			if (p_Command->p_Word_Param4 != 0xFFFF)
			{
				float l_Temp = 10000.0 * (((float) p_Command->p_Word_Param4) / 65534.0);
				f_Level->Set_Release (l_Temp);
			}

			break;
		}
	case 0x83 :
		{
			//  Set Pitch Bend Rate

			if (p_Command->p_Word_Param1 != 0xFFFF)
			{
				float l_Temp = 10000.0 * (((float) p_Command->p_Word_Param1) / 65534.0);
				f_Pitch->Set_Attack (l_Temp);
			}

			break;
		}
	case 0x84 :
		{
			//  Set Twin Note

			if (p_Command->p_Byte_Param1 != 0xFF)
			{
				f_Enable_Twin = (p_Command->p_Byte_Param1 != 0);
			}

			if (p_Command->p_Byte_Param2 != 0xFF)
			{
				f_Fractal_Before = (p_Command->p_Byte_Param2 != 0);
			}

			if (p_Command->p_Word_Param2 != 0xFFFF)
			{
				f_Twin_Amp = (((float) p_Command->p_Word_Param1) / 65534.0);
			}

			break;
		}
	case 0x85 :
		{
			//  Set Ring Modulations

			if (p_Command->p_Byte_Param1 != 0xFF)
			{
				f_Ringmod_Main = (p_Command->p_Byte_Param1 != 0);
			}

			if (p_Command->p_Byte_Param2 != 0xFF)
			{
				f_Ringmod_Twin = (p_Command->p_Byte_Param2 != 0);
			}

			break;
		}
	case 0x86 :
		{
			//  Set Frequency Modulation

			if (p_Command->p_Byte_Param1 != 0xFF)
			{
				f_FM_Enable = (p_Command->p_Byte_Param1 != 0);
			}

			if (p_Command->p_Byte_Param2 != 0xFF)
			{
				f_FM_Internal = (p_Command->p_Byte_Param2 != 0);
			}

			if (p_Command->p_Word_Param1 != 0xFFFF)
			{
				f_FM_Level = 10.0 * (((float) p_Command->p_Word_Param1) / 65534.0);
			}

			break;
		}
	case 0x87 :
		{
			//  Set Amplitude Modulation

			if (p_Command->p_Byte_Param1 != 0xFF)
			{
				f_AM_Enable = (p_Command->p_Byte_Param1 != 0);
			}

			if (p_Command->p_Byte_Param2 != 0xFF)
			{
				f_AM_Internal = (p_Command->p_Byte_Param2 != 0);
			}

			if (p_Command->p_Word_Param1 != 0xFFFF)
			{
				f_AM_Level = (((float) p_Command->p_Word_Param1) / 65534.0);
			}

			break;
		}
	case 0x88 :
		{
			//  Set Pulse Width Modulation

			if (p_Command->p_Byte_Param1 != 0xFF)
			{
				f_PWM_Enable = (p_Command->p_Byte_Param1 != 0);
			}

			if (p_Command->p_Byte_Param2 != 0xFF)
			{
				f_PWM_Internal = (p_Command->p_Byte_Param2 != 0);
			}

			if (p_Command->p_Word_Param1 != 0xFFFF)
			{
				f_PWM_Level = (((float) p_Command->p_Word_Param1) / 65534.0);
			}

			break;
		}
	case 0x89 :
		{
			//  Set modulator waveform from p_Byte_Param1

			if (   (p_Command->p_Byte_Param1 >= 1                   )
				&& (p_Command->p_Byte_Param1 <= NR_LIB_HIGH_WAVEFORM))
			{
				f_Mod_Waveform->Set_Waveform (p_Command->p_Byte_Param1);
			}

			break;
		}
	case 0x8A :
		{
			//  Set modulator fractal

			if (   (p_Command->p_Byte_Param1 >=  0)
				&& (p_Command->p_Byte_Param1 <= 10))
			{
				f_Mod_Fractal->Set_Depth (p_Command->p_Byte_Param1);
			}

			if (p_Command->p_Word_Param1 != 0xFFFF)
			{
				float l_Effect = 9.0 * (((float) p_Command->p_Word_Param1) / 65534.0);
				f_Mod_Fractal->Set_Effect_Lo (l_Effect);
			}

			if (p_Command->p_Word_Param2 != 0xFFFF)
			{
				float l_Effect = 9.0 * (((float) p_Command->p_Word_Param2) / 65534.0);
				f_Mod_Fractal->Set_Effect_Hi (l_Effect);
			}

			break;
		}
	}
}

void CTrack::Reset()
{
	//  Should probably reset the tracks envelope here
}

void CTrack::Tick(tvals const &tv)
{
	if (tv.volume != paraVolume.NoValue)
	{
		f_Level->Set_Volume (((float) tv.volume) / ((float) 0x80));
	}

	if (tv.noteinit == NOTE_OFF)
	{
		f_Level->Stop_Note     ();
		f_Mod_Level->Stop_Note ();
	}
	else if (tv.noteinit != NOTE_NO)
	{
		f_Level->Start_Note     ();
		f_Mod_Level->Start_Note ();
	}

	f_Pitch->Tick (tv.note,     tv.noteinit);
}

void CTrack::Stop()
{
	f_Level->Stop     ();
	f_Mod_Level->Stop ();
	f_Pitch->Stop     ();
}

//  Apply_Std_Mod handles amplitude and frequency modulation, depending on the
//  contents of the buffer it is applied to.

void Apply_Std_Mod (float *p_Dest, float *p_Src, int p_Num_Samples, float p_Level)
{
  if (p_Level != 0.0)
  {
    if (p_Level == 1.0)
    {
      while (p_Num_Samples-- > 0)
      {
        *(p_Dest++) *= (*(p_Src++) + 1.0);
      }
    }
    else
    {
      while (p_Num_Samples-- > 0)
      {
        *(p_Dest++) *= ((*(p_Src++) * p_Level) + 1.0);
      }
    }
  }
}

//  PWM needs a different algorithm

void Apply_PWM (float *p_Dest, float *p_Src, int p_Num_Samples, float p_Level)
{
  if (p_Level != 0.0)
  {
    if (p_Level == 1.0)
    {
      while (p_Num_Samples-- > 0)
			{
				if (*p_Dest < PI)
				{
					*p_Dest *= (*p_Src + 1.0) * 0.5;
				}
				else
				{
					*p_Dest = ((*p_Dest - 1.0) * (1.0 - (*p_Src)) * 0.5) + 1.0;
				}

				p_Dest++; p_Src++;
			}
    }
    else
    {
      while (p_Num_Samples-- > 0)
			{
				if (*p_Dest < PI)
				{
					*p_Dest *= ((*p_Src * p_Level) + 1.0) * 0.5;
				}
				else
				{
					*p_Dest = ((*p_Dest - 1.0) * (1.0 - (*p_Src * -p_Level)) * 0.5) + 1.0;
				}

				p_Dest++; p_Src++;
			}
    }
  }
}

//
//  The following Generate function is starting to accumulate special cases
//
//  For readability (and possibly a small effeciency gain) it may be better
//  to have several generate functions.
//

bool CTrack::Generate(float *psamples, int numsamples)
{
	//  Derive envelope

	if (f_Level->Is_Static ())
	{
		if (f_Level->Current_Level () == 0.0)
		{
			return false;
		}
	}

  //  Evaluate and save repeat conditions

  bool l_Mod_Level_Zero = (f_Mod_Level->Is_Static () && (f_Mod_Level->Current_Level () == 0.0));

  bool l_FM_Internal  = (pmi->f_FM_Enable  && pmi->f_FM_Internal  && (!l_Mod_Level_Zero));
  bool l_AM_Internal  = (pmi->f_AM_Enable  && pmi->f_AM_Internal  && (!l_Mod_Level_Zero));
  bool l_PWM_Internal = (pmi->f_PWM_Enable && pmi->f_PWM_Internal && (!l_Mod_Level_Zero));

  bool l_FM_Needed    = (pmi->f_FM_Enable  && ((pmi->f_FM_Internal  && (!l_Mod_Level_Zero)) || (pmi->f_Channel != -1)));
  bool l_AM_Needed    = (pmi->f_AM_Enable  && ((pmi->f_AM_Internal  && (!l_Mod_Level_Zero)) || (pmi->f_Channel != -1)));
  bool l_PWM_Needed   = (pmi->f_PWM_Enable && ((pmi->f_PWM_Internal && (!l_Mod_Level_Zero)) || (pmi->f_Channel != -1)));

  bool l_Twin_Needed        = pmi->f_Enable_Twin;
  bool l_Twin_Mix_First     = l_Twin_Needed && (!pmi->f_Fractal_Before);
  bool l_Twin_Fractal_First = l_Twin_Needed && ( pmi->f_Fractal_Before);

  bool l_Ringmod_Main_Needed = pmi->f_Ringmod_Main && (pmi->f_Channel != -1);
  bool l_Ringmod_Twin_Needed = pmi->f_Ringmod_Twin && (pmi->f_Channel != -1) && l_Twin_Needed;

	//  Both pitch and amplitude envelopes might benefit from more frequent static
	//  checks, allowing an envelope to become static in the middle of a block. However,
	//  tuning of the minimum amplitude might be better.
    
	f_Pitch->Overwrite (psamples, numsamples);

  //  Derive modulator - this is quick and dirty at present - much more optimisation needed

  if (l_FM_Internal || l_AM_Internal || l_PWM_Internal)
  {
    float l_Amp = 1.0;

    switch (pmi->f_Mod_Frq_Type)
    {
      case 0 :  l_Amp = pow (2.0,  pmi->f_Mod_Frq       );  break;
      case 1 :  l_Amp = pow (2.0, -pmi->f_Mod_Frq       );  break;
      case 2 :  l_Amp = pow (2.0,  pmi->f_Mod_Frq / 12.0);  break;
      case 3 :  l_Amp = pow (2.0, -pmi->f_Mod_Frq / 12.0);  break;
      case 4 :  l_Amp = pmi->f_Mod_Frq;                     break;
      case 5 :  if (pmi->f_Mod_Frq > 0.0)  {  l_Amp = 1.0 / pmi->f_Mod_Frq;  }  break;
    }

    if (l_Mod_Level_Zero)
    {
      DSP_Zero (f_Mod_Buffer, numsamples);
    }
    else
    {
      if (f_Mod_Level->Is_Static ())  //  Optimise fractal and envelope for constant envelope level
      {
        f_Mod_Frq_To_Phase->Process   (f_Mod_Buffer, psamples, numsamples, l_Amp);
        f_Mod_Waveform->Process       (f_Mod_Buffer, numsamples);
        f_Mod_Fractal->Process_Static (f_Mod_Buffer, f_Mod_Level->Current_Level (), numsamples);
        f_Mod_Level->Multiply         (f_Buffer2, numsamples);
      }
      else
      {
        f_Mod_Level->Overwrite (f_Buffer2, numsamples);

        f_Mod_Frq_To_Phase->Process (f_Mod_Buffer, psamples, numsamples, l_Amp);
        f_Mod_Waveform->Process     (f_Mod_Buffer, numsamples);
        f_Mod_Fractal->Process      (f_Mod_Buffer, f_Buffer2, numsamples);
        NR_DSP_Multiply             (f_Mod_Buffer, f_Buffer2, 32768.0, numsamples);
      }
    }
  }

	if (l_FM_Needed)
	{
    float *b;

    if (pmi->f_FM_Internal)
    {
      b = f_Mod_Buffer;
    }
    else
    {
      b = pmi->f_Aux_Buffer;
    }

    Apply_Std_Mod (psamples, b, numsamples, pmi->f_FM_Level / 32768.0);
	}

	if (l_Twin_Needed)
	{
		f_Frq_To_Phase2->Process (f_Buffer2, psamples, numsamples, pow(2.0, pmi->f_Detune_Semitones / 12.0));

		if (l_PWM_Needed)
		{
      float *b;

      if (pmi->f_PWM_Internal)
      {
        b = f_Mod_Buffer;
      }
      else
      {
        b = pmi->f_Aux_Buffer;
      }

      Apply_PWM (f_Buffer2, b, numsamples, pmi->f_PWM_Level / 32768.0);
		}
		
		f_Waveform->Process      (f_Buffer2, numsamples);

		if (l_Ringmod_Twin_Needed)
		{
			NR_DSP_Multiply (f_Buffer2, pmi->f_Aux_Buffer, 1.0 / 32768.0, numsamples);
		}
	}

	f_Frq_To_Phase->Process (psamples, numsamples);

	if (l_PWM_Needed)
	{
		float *b;

    if (pmi->f_PWM_Internal)
    {
      b = f_Mod_Buffer;
    }
    else
    {
      b = pmi->f_Aux_Buffer;
    }

    Apply_PWM (psamples, b, numsamples, pmi->f_PWM_Level / 32768.0);
	}

	f_Waveform->Process     (psamples, numsamples);

	if (l_Ringmod_Main_Needed)
	{
		NR_DSP_Multiply (psamples, pmi->f_Aux_Buffer, 1.0 / 32768.0, numsamples);
	}

	if (l_Twin_Mix_First)
	{
		DSP_Add (psamples, f_Buffer2, numsamples, pmi->f_Twin_Amp);
		DSP_Amp (psamples, numsamples, 0.5);       //  Scaling necessary for fractal to work
	}

	//  Apply envelope and Fractal, and normalise amplitude

	if (f_Level->Is_Static ())
	{
		float l_Level = f_Level->Current_Level ();

		f_Fractal->Process_Static (psamples,  l_Level, numsamples);

		if (l_Twin_Fractal_First)
		{
			f_Fractal->Process_Static (f_Buffer2, l_Level, numsamples);

			DSP_Add (psamples, f_Buffer2, numsamples, pmi->f_Twin_Amp);
			DSP_Amp (psamples, numsamples, l_Level * 16384.0);
		}
		else
		{
			DSP_Amp (psamples, numsamples, l_Level * 32768.0);
		}
	}
	else
	{
		f_Level->Overwrite (f_Buffer, numsamples);

		if (f_Fractal->Is_Static ())
		{
			//  If the fractal is static the modulator is irrelevant, so it may as well be zero
			f_Fractal->Process_Static (psamples,  0.0, numsamples);

			if (l_Twin_Fractal_First)
			{
				f_Fractal->Process_Static (f_Buffer2, 0.0, numsamples);
			}
		}
		else
		{
			f_Fractal->Process (psamples,  f_Buffer, numsamples);

			if (l_Twin_Fractal_First)
			{
				f_Fractal->Process (f_Buffer2, f_Buffer, numsamples);
			}
		}

		if (l_Twin_Fractal_First)
		{
			DSP_Add (psamples, f_Buffer2, numsamples, pmi->f_Twin_Amp);

			NR_DSP_Multiply (psamples, f_Buffer, (float) 16384.0, numsamples);
		}
		else
		{
			NR_DSP_Multiply (psamples, f_Buffer, (float) 32768.0, numsamples);
		}
	}

	if (l_AM_Needed)
	{
    float *b;

    if (pmi->f_AM_Internal)
    {
      b = f_Mod_Buffer;
    }
    else
    {
      b = pmi->f_Aux_Buffer;
    }

    Apply_Std_Mod (psamples, b, numsamples, pmi->f_AM_Level / 32768.0);
	}

	return true;
}

HINSTANCE dllInstance;

mi   *g_mi;

HWND g_Propsheet_HWnd [6];

BOOL WINAPI DllMain( HANDLE hModule, DWORD fdwreason, LPVOID lpReserved)
{
    switch(fdwreason)
	{
    case DLL_PROCESS_ATTACH:
	{
		// The DLL is being mapped into process's address space
		//  Do any required initialization on a per application basis, return FALSE if failed
		dllInstance=(HINSTANCE) hModule;

		g_mi = NULL;

		for (int i = 0; i < 6; i++)
		{
			g_Propsheet_HWnd [i] = NULL;
		}

		break;
	}
    case DLL_THREAD_ATTACH:
		// A thread is created. Do any required initialization on a per thread basis
		break;
    case DLL_THREAD_DETACH:
		// Thread exits with  cleanup
		break;
    case DLL_PROCESS_DETACH:
	    // The DLL unmapped from process's address space. Do necessary cleanup
		break;
    }
    
	return TRUE;
}

#define M_CHECK_RADIO(p)  CheckRadioButton (hDlg, IDC_SINE, IDC_DBL_TRIANGLE2, p)

#define M_FROM_SLIDER(p) pow(10.0, (((float) (p)) / 3000.0) + (2.0 / 3.0))
#define M_TO_SLIDER(p)   ((log10((float) (p)) - (2.0 / 3.0)) * 3000.0)

//
//  Control handling macros, to make it easier to move stuff from one page to
//  another or to have duplicate pages for similar types of data...
//

//  Logarithmic scale sliders range 1.0..10000.0 : Edit control notify

#define M_LOG_SLIDER_EDIT_NOTIFY(p_Slider_ID,p_Edit_ID,p_Dest)                                    \
			case p_Edit_ID :                                                                            \
			{                                                                                           \
				char  l_Buf [80];                                                                         \
                                                                                                  \
				GetDlgItemText (hDlg, p_Edit_ID, l_Buf, 32);                                              \
                                                                                                  \
				float l_Temp = atof (l_Buf) * 1000.0;                                                     \
                                                                                                  \
				if ((l_Temp >= 1.0) && (l_Temp <= 10000.0))                                               \
				{                                                                                         \
					p_Dest;                                                                                 \
					SendDlgItemMessage (hDlg, p_Slider_ID, TBM_SETPOS, TRUE, M_TO_SLIDER (l_Temp));         \
				}                                                                                         \
                                                                                                  \
				return 1;                                                                                 \
			}

//  Slider notify

#define M_LOG_SLIDER_SLIDE_NOTIFY(p_Slider_ID,p_Edit_ID,p_Dest)                                    \
		if ((HWND) lParam == GetDlgItem (hDlg, p_Slider_ID))                                           \
		{                                                                                              \
			float l_Temp = M_FROM_SLIDER (SendDlgItemMessage (hDlg, p_Slider_ID, TBM_GETPOS, 0, 0));     \
                                                                                                   \
			p_Dest;                                                                                      \
                                                                                                   \
			char  l_Buf [80];                                                                            \
                                                                                                   \
			sprintf (l_Buf, "%f", l_Temp / 1000.0);  SetDlgItemText (hDlg, p_Edit_ID, l_Buf);            \
		}


//  Linear scale sliders range 0.0..1.0 : Edit control notify

#define M_LIN_SLIDER_EDIT_NOTIFY(p_Slider_ID,p_Edit_ID,p_Dest)                                    \
			case p_Edit_ID :                                                                            \
			{                                                                                           \
				char  l_Buf [80];                                                                         \
                                                                                                  \
				GetDlgItemText (hDlg, p_Edit_ID, l_Buf, 32);                                              \
                                                                                                  \
				float l_Temp = atof (l_Buf);                                                              \
                                                                                                  \
				if ((l_Temp >= 0.0) && (l_Temp <= 1.0))                                                   \
				{                                                                                         \
					p_Dest;                                                                                 \
					SendDlgItemMessage (hDlg, p_Slider_ID, TBM_SETPOS, TRUE, (int) (l_Temp * 10000.0));     \
				}                                                                                         \
                                                                                                  \
				return 1;                                                                                 \
			}

//  Slider notify

#define M_LIN_SLIDER_SLIDE_NOTIFY(p_Slider_ID,p_Edit_ID,p_Dest)                                    \
		if ((HWND) lParam == GetDlgItem (hDlg, p_Slider_ID))                                           \
		{                                                                                              \
			float l_Temp = ((float) SendDlgItemMessage (hDlg, p_Slider_ID, TBM_GETPOS, 0, 0)) / 10000.0; \
                                                                                                   \
			p_Dest;                                                                                      \
                                                                                                   \
			char  l_Buf [80];                                                                            \
                                                                                                   \
			sprintf (l_Buf, "%f", l_Temp);  SetDlgItemText (hDlg, p_Edit_ID, l_Buf);                     \
		}




void Update_Waveform_Grp (HWND hDlg, c_Waveform *p_Waveform)
{
	switch (p_Waveform->Get_Waveform ())
	{
	case  1 :  M_CHECK_RADIO (IDC_SINE         );  break;
	case  2 :  M_CHECK_RADIO (IDC_TRIANGLE     );  break;
	case  3 :  M_CHECK_RADIO (IDC_DBL_TRIANGLE );  break;
	case  4 :  M_CHECK_RADIO (IDC_HEX          );  break;
	case  5 :  M_CHECK_RADIO (IDC_SQUARE       );  break;
	case  6 :  M_CHECK_RADIO (IDC_RAMP         );  break;
	case  7 :  M_CHECK_RADIO (IDC_ALT_RAMP     );  break;
	case  8 :  M_CHECK_RADIO (IDC_BUMPS        );  break;
	case  9 :  M_CHECK_RADIO (IDC_DBL_BUMPS    );  break;
	case 10 :  M_CHECK_RADIO (IDC_ALT_BUMPS    );  break;
	case 11 :  M_CHECK_RADIO (IDC_ALT_BUMPS2   );  break;
	case 12 :  M_CHECK_RADIO (IDC_SQUARE_BUMPS );  break;
	case 13 :  M_CHECK_RADIO (IDC_DBL_BUMPS2   );  break;
	case 14 :  M_CHECK_RADIO (IDC_DBL_TRIANGLE2);  break;
	}
}

void Update_Waveform_Dlg (mi *pmi, HWND hDlg)
{
  Update_Waveform_Grp (hDlg, pmi->f_Waveform);

	CheckDlgButton (hDlg, IDC_CHECK_ENABLE_TWIN,    pmi->f_Enable_Twin   );
	CheckDlgButton (hDlg, IDC_CHECK_FRACTAL_BEFORE, pmi->f_Fractal_Before);

	SendDlgItemMessage (hDlg, IDC_SLIDER_DEPTH,     TBM_SETRANGE, TRUE, MAKELONG(0,   10));
	SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_HI, TBM_SETRANGE, TRUE, MAKELONG(0, 9000));
	SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_LO, TBM_SETRANGE, TRUE, MAKELONG(0, 9000));
	SendDlgItemMessage (hDlg, IDC_SLIDER_DETUNE,    TBM_SETRANGE, TRUE, MAKELONG(-10000, 10000));
	SendDlgItemMessage (hDlg, IDC_SLIDER_TWIN_AMP,  TBM_SETRANGE, TRUE, MAKELONG(0,      10000));

	{
		int   l_Depth     = pmi->f_Fractal->Get_Depth ();
		float l_Effect_Hi = pmi->f_Fractal->Get_Effect_Hi ();
		float l_Effect_Lo = pmi->f_Fractal->Get_Effect_Lo ();

		SendDlgItemMessage (hDlg, IDC_SLIDER_DEPTH,     TBM_SETPOS, TRUE, l_Depth             );
		SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_HI, TBM_SETPOS, TRUE, l_Effect_Hi * 1000.0);
		SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_LO, TBM_SETPOS, TRUE, l_Effect_Lo * 1000.0);
		SendDlgItemMessage (hDlg, IDC_SLIDER_DETUNE,    TBM_SETPOS, TRUE, pmi->f_Detune_Semitones * 10000.0 / 12.0);
		SendDlgItemMessage (hDlg, IDC_SLIDER_TWIN_AMP,  TBM_SETPOS, TRUE, pmi->f_Twin_Amp * 10000.0);

		char  l_Buf [80];

		sprintf (l_Buf, "%d", l_Depth    );  SetDlgItemText (hDlg, IDC_EDIT_DEPTH,     l_Buf);
		sprintf (l_Buf, "%f", l_Effect_Hi);  SetDlgItemText (hDlg, IDC_EDIT_EFFECT_HI, l_Buf);
		sprintf (l_Buf, "%f", l_Effect_Lo);  SetDlgItemText (hDlg, IDC_EDIT_EFFECT_LO, l_Buf);
		sprintf (l_Buf, "%f", pmi->f_Detune_Semitones);  SetDlgItemText (hDlg, IDC_EDIT_DETUNE, l_Buf);
		sprintf (l_Buf, "%f", pmi->f_Twin_Amp);  SetDlgItemText (hDlg, IDC_EDIT_TWIN_AMP, l_Buf);
	}
}

void Update_Mod_Waveform_Dlg (mi *pmi, HWND hDlg)
{
  Update_Waveform_Grp (hDlg, pmi->f_Mod_Waveform);

	SendDlgItemMessage (hDlg, IDC_SLIDER_DEPTH,     TBM_SETRANGE, TRUE, MAKELONG(0,    10));
	SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_HI, TBM_SETRANGE, TRUE, MAKELONG(0,  9000));
	SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_LO, TBM_SETRANGE, TRUE, MAKELONG(0,  9000));
	SendDlgItemMessage (hDlg, IDC_SLIDER_MODFRQ,    TBM_SETRANGE, TRUE, MAKELONG(0, 24000));

	{
		int   l_Depth     = pmi->f_Mod_Fractal->Get_Depth ();
		float l_Effect_Hi = pmi->f_Mod_Fractal->Get_Effect_Hi ();
		float l_Effect_Lo = pmi->f_Mod_Fractal->Get_Effect_Lo ();

		SendDlgItemMessage (hDlg, IDC_SLIDER_DEPTH,     TBM_SETPOS, TRUE, l_Depth                );
		SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_HI, TBM_SETPOS, TRUE, l_Effect_Hi    * 1000.0);
		SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_LO, TBM_SETPOS, TRUE, l_Effect_Lo    * 1000.0);
		SendDlgItemMessage (hDlg, IDC_SLIDER_MODFRQ,    TBM_SETPOS, TRUE, pmi->f_Mod_Frq * 1000.0);

		char  l_Buf [80];

		sprintf (l_Buf, "%d", l_Depth       );  SetDlgItemText (hDlg, IDC_EDIT_DEPTH,     l_Buf);
		sprintf (l_Buf, "%f", l_Effect_Hi   );  SetDlgItemText (hDlg, IDC_EDIT_EFFECT_HI, l_Buf);
		sprintf (l_Buf, "%f", l_Effect_Lo   );  SetDlgItemText (hDlg, IDC_EDIT_EFFECT_LO, l_Buf);
		sprintf (l_Buf, "%f", pmi->f_Mod_Frq);  SetDlgItemText (hDlg, IDC_EDIT_MODFRQ,    l_Buf);
	}

  switch (pmi->f_Mod_Frq_Type)
  {
    case 0 : CheckRadioButton (hDlg, IDC_RADIO_RAISE_OCTAVES, IDC_RADIO_DIVIDE_FRQ, IDC_RADIO_RAISE_OCTAVES  );  break;
    case 1 : CheckRadioButton (hDlg, IDC_RADIO_RAISE_OCTAVES, IDC_RADIO_DIVIDE_FRQ, IDC_RADIO_LOWER_OCTAVES  );  break;
    case 2 : CheckRadioButton (hDlg, IDC_RADIO_RAISE_OCTAVES, IDC_RADIO_DIVIDE_FRQ, IDC_RADIO_RAISE_SEMITONES);  break;
    case 3 : CheckRadioButton (hDlg, IDC_RADIO_RAISE_OCTAVES, IDC_RADIO_DIVIDE_FRQ, IDC_RADIO_LOWER_SEMITONES);  break;
    case 4 : CheckRadioButton (hDlg, IDC_RADIO_RAISE_OCTAVES, IDC_RADIO_DIVIDE_FRQ, IDC_RADIO_MULTIPLY_FRQ   );  break;
    case 5 : CheckRadioButton (hDlg, IDC_RADIO_RAISE_OCTAVES, IDC_RADIO_DIVIDE_FRQ, IDC_RADIO_DIVIDE_FRQ     );  break;
  }
}

void Update_Aux_Bus_Dlg (mi *pmi, HWND hDlg)
{
	CheckDlgButton (hDlg, IDC_CHECK_RINGMOD_MAIN, pmi->f_Ringmod_Main);
	CheckDlgButton (hDlg, IDC_CHECK_RINGMOD_TWIN, pmi->f_Ringmod_Twin);
	CheckDlgButton (hDlg, IDC_CHECK_FM_ENABLE,    pmi->f_FM_Enable );
	CheckDlgButton (hDlg, IDC_CHECK_AM_ENABLE,    pmi->f_AM_Enable );
	CheckDlgButton (hDlg, IDC_CHECK_PWM_ENABLE,   pmi->f_PWM_Enable);
	CheckDlgButton (hDlg, IDC_CHECK_FM_INTERNAL,  pmi->f_FM_Internal);
	CheckDlgButton (hDlg, IDC_CHECK_AM_INTERNAL,  pmi->f_AM_Internal );
	CheckDlgButton (hDlg, IDC_CHECK_PWM_INTERNAL, pmi->f_PWM_Internal);
  CheckDlgButton (hDlg, IDC_CHECK_WAV_OPT,      pmi->f_Waveform->Get_Optimise_Enable ());
  CheckDlgButton (hDlg, IDC_CHECK_FRACT_OPT,    pmi->f_Fractal->Get_Optimise_Enable ());

	SendDlgItemMessage (hDlg, IDC_SLIDER_FM_LEVEL,  TBM_SETRANGE, TRUE, MAKELONG(0,      10000));
	SendDlgItemMessage (hDlg, IDC_SLIDER_AM_LEVEL,  TBM_SETRANGE, TRUE, MAKELONG(0,      10000));
	SendDlgItemMessage (hDlg, IDC_SLIDER_PWM_LEVEL, TBM_SETRANGE, TRUE, MAKELONG(0,      10000));

	{
		SendDlgItemMessage (hDlg, IDC_SLIDER_FM_LEVEL,  TBM_SETPOS, TRUE, M_TO_SLIDER(pmi->f_FM_Level  *  1000.0));
		SendDlgItemMessage (hDlg, IDC_SLIDER_AM_LEVEL,  TBM_SETPOS, TRUE, M_TO_SLIDER(pmi->f_AM_Level  * 10000.0));
		SendDlgItemMessage (hDlg, IDC_SLIDER_PWM_LEVEL, TBM_SETPOS, TRUE, M_TO_SLIDER(pmi->f_PWM_Level * 10000.0));

		char  l_Buf [80];

		sprintf (l_Buf, "%f", pmi->f_FM_Level);  SetDlgItemText (hDlg, IDC_EDIT_FM_LEVEL,  l_Buf);
		sprintf (l_Buf, "%f", pmi->f_AM_Level);  SetDlgItemText (hDlg, IDC_EDIT_AM_LEVEL,  l_Buf);
		sprintf (l_Buf, "%f", pmi->f_PWM_Level); SetDlgItemText (hDlg, IDC_EDIT_PWM_LEVEL, l_Buf);
    sprintf (l_Buf, "%d", pmi->f_Waveform->Get_Optimise_Quality ());  SetDlgItemText (hDlg, IDC_EDIT_WAV_OPT,  l_Buf);
    sprintf (l_Buf, "%d", pmi->f_Fractal->Get_Optimise_Quality ());  SetDlgItemText (hDlg, IDC_EDIT_FRACT_OPT,  l_Buf);
	}
}

void Update_Envel_Dlg (mi *pmi, HWND hDlg)
{
	SendDlgItemMessage (hDlg, IDC_SLIDER_ATTACK,  TBM_SETRANGE, TRUE, MAKELONG(0, 10000));
	SendDlgItemMessage (hDlg, IDC_SLIDER_DECAY,   TBM_SETRANGE, TRUE, MAKELONG(0, 10000));
	SendDlgItemMessage (hDlg, IDC_SLIDER_SUSTAIN, TBM_SETRANGE, TRUE, MAKELONG(0, 10000));
	SendDlgItemMessage (hDlg, IDC_SLIDER_RELEASE, TBM_SETRANGE, TRUE, MAKELONG(0, 10000));
	SendDlgItemMessage (hDlg, IDC_SLIDER_BEND,    TBM_SETRANGE, TRUE, MAKELONG(0, 10000));

	{
		float l_Attack  = pmi->f_Level->Get_Attack  ();
		float l_Decay   = pmi->f_Level->Get_Decay   ();
		float l_Sustain = pmi->f_Level->Get_Sustain ();
		float l_Release = pmi->f_Level->Get_Release ();
		float l_Bend    = pmi->f_Pitch->Get_Attack  ();

		SendDlgItemMessage (hDlg, IDC_SLIDER_ATTACK,  TBM_SETPOS, TRUE, M_TO_SLIDER (l_Attack) );
		SendDlgItemMessage (hDlg, IDC_SLIDER_DECAY,   TBM_SETPOS, TRUE, M_TO_SLIDER (l_Decay ) );
		SendDlgItemMessage (hDlg, IDC_SLIDER_SUSTAIN, TBM_SETPOS, TRUE, l_Sustain *  10000.0   );
		SendDlgItemMessage (hDlg, IDC_SLIDER_RELEASE, TBM_SETPOS, TRUE, M_TO_SLIDER (l_Release));
		SendDlgItemMessage (hDlg, IDC_SLIDER_BEND,    TBM_SETPOS, TRUE, M_TO_SLIDER (l_Bend   ));

		char  l_Buf [80];

		sprintf (l_Buf, "%f", l_Attack  / 1000.0);  SetDlgItemText (hDlg, IDC_EDIT_ATTACK,  l_Buf);
		sprintf (l_Buf, "%f", l_Decay   / 1000.0);  SetDlgItemText (hDlg, IDC_EDIT_DECAY,   l_Buf);
		sprintf (l_Buf, "%f", l_Sustain         );  SetDlgItemText (hDlg, IDC_EDIT_SUSTAIN, l_Buf);
		sprintf (l_Buf, "%f", l_Release / 1000.0);  SetDlgItemText (hDlg, IDC_EDIT_RELEASE, l_Buf);
		sprintf (l_Buf, "%f", l_Bend    / 1000.0);  SetDlgItemText (hDlg, IDC_EDIT_BEND,    l_Buf);
	}
}

void Update_Mod_Envel_Dlg (mi *pmi, HWND hDlg)
{
	SendDlgItemMessage (hDlg, IDC_SLIDER_ATTACK,  TBM_SETRANGE, TRUE, MAKELONG(0, 10000));
	SendDlgItemMessage (hDlg, IDC_SLIDER_DECAY,   TBM_SETRANGE, TRUE, MAKELONG(0, 10000));
	SendDlgItemMessage (hDlg, IDC_SLIDER_SUSTAIN, TBM_SETRANGE, TRUE, MAKELONG(0, 10000));
	SendDlgItemMessage (hDlg, IDC_SLIDER_RELEASE, TBM_SETRANGE, TRUE, MAKELONG(0, 10000));
	SendDlgItemMessage (hDlg, IDC_SLIDER_START,   TBM_SETRANGE, TRUE, MAKELONG(0, 10000));
	SendDlgItemMessage (hDlg, IDC_SLIDER_PEAK,    TBM_SETRANGE, TRUE, MAKELONG(0, 10000));
	SendDlgItemMessage (hDlg, IDC_SLIDER_END,     TBM_SETRANGE, TRUE, MAKELONG(0, 10000));

	{
		float l_Attack  = pmi->f_Mod_Level->Get_Attack  ();
		float l_Decay   = pmi->f_Mod_Level->Get_Decay   ();
		float l_Sustain = pmi->f_Mod_Level->Get_Sustain ();
		float l_Release = pmi->f_Mod_Level->Get_Release ();
		float l_Start   = pmi->f_Mod_Level->Get_Start   ();
		float l_Peak    = pmi->f_Mod_Level->Get_Peak    ();
		float l_End     = pmi->f_Mod_Level->Get_End     ();

		SendDlgItemMessage (hDlg, IDC_SLIDER_ATTACK,  TBM_SETPOS, TRUE, M_TO_SLIDER (l_Attack) );
		SendDlgItemMessage (hDlg, IDC_SLIDER_DECAY,   TBM_SETPOS, TRUE, M_TO_SLIDER (l_Decay ) );
		SendDlgItemMessage (hDlg, IDC_SLIDER_SUSTAIN, TBM_SETPOS, TRUE, l_Sustain *  10000.0   );
		SendDlgItemMessage (hDlg, IDC_SLIDER_RELEASE, TBM_SETPOS, TRUE, M_TO_SLIDER (l_Release));
		SendDlgItemMessage (hDlg, IDC_SLIDER_START,   TBM_SETPOS, TRUE, l_Start   *  10000.0   );
		SendDlgItemMessage (hDlg, IDC_SLIDER_PEAK,    TBM_SETPOS, TRUE, l_Peak    *  10000.0   );
		SendDlgItemMessage (hDlg, IDC_SLIDER_END,     TBM_SETPOS, TRUE, l_End     *  10000.0   );

		char  l_Buf [80];

		sprintf (l_Buf, "%f", l_Attack  / 1000.0);  SetDlgItemText (hDlg, IDC_EDIT_ATTACK,  l_Buf);
		sprintf (l_Buf, "%f", l_Decay   / 1000.0);  SetDlgItemText (hDlg, IDC_EDIT_DECAY,   l_Buf);
		sprintf (l_Buf, "%f", l_Sustain         );  SetDlgItemText (hDlg, IDC_EDIT_SUSTAIN, l_Buf);
		sprintf (l_Buf, "%f", l_Release / 1000.0);  SetDlgItemText (hDlg, IDC_EDIT_RELEASE, l_Buf);
		sprintf (l_Buf, "%f", l_Start           );  SetDlgItemText (hDlg, IDC_EDIT_START,   l_Buf);
		sprintf (l_Buf, "%f", l_Peak            );  SetDlgItemText (hDlg, IDC_EDIT_PEAK,    l_Buf);
		sprintf (l_Buf, "%f", l_End             );  SetDlgItemText (hDlg, IDC_EDIT_END,     l_Buf);
	}
}

void Update_All_Dialogs (void)  //  Exclude the Rx Broadcast page
{
	if (g_Propsheet_HWnd [0] != NULL)
	{
		Update_Waveform_Dlg (g_mi, g_Propsheet_HWnd [0]);
	}

	if (g_Propsheet_HWnd [1] != NULL)
	{
		Update_Aux_Bus_Dlg (g_mi, g_Propsheet_HWnd [1]);
	}

	if (g_Propsheet_HWnd [2] != NULL)
	{
		Update_Envel_Dlg (g_mi, g_Propsheet_HWnd [2]);
	}

	if (g_Propsheet_HWnd [3] != NULL)
	{
		Update_Mod_Waveform_Dlg (g_mi, g_Propsheet_HWnd [3]);
	}

	if (g_Propsheet_HWnd [4] != NULL)
	{
		Update_Mod_Envel_Dlg (g_mi, g_Propsheet_HWnd [4]);
	}
}

BOOL APIENTRY MainDialog (HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
	case WM_INITDIALOG :
		g_Propsheet_HWnd [0] = hDlg;

		Update_Waveform_Dlg (g_mi, hDlg);

		return 1;

	case WM_COMMAND :
		switch (LOWORD (wParam))
		{
			//  The CheckRadioButton calls here shouldn't be necessary, but maybe I've hit some
			//  obscure limit.
			//
			//  It might be better to use a combobox or listbox - it will certainly be easier to
			//  extend in the future, and could save space on screen when multiple oscilators are
			//  added (modulators etc).
	
		    case IDC_SINE          :  M_CHECK_RADIO (IDC_SINE         );  g_mi->f_Waveform->Set_Waveform ( 1);  return 1;
			case IDC_TRIANGLE      :  M_CHECK_RADIO (IDC_TRIANGLE     );  g_mi->f_Waveform->Set_Waveform ( 2);  return 1;
			case IDC_DBL_TRIANGLE  :  M_CHECK_RADIO (IDC_DBL_TRIANGLE );  g_mi->f_Waveform->Set_Waveform ( 3);  return 1;
			case IDC_HEX           :  M_CHECK_RADIO (IDC_HEX          );  g_mi->f_Waveform->Set_Waveform ( 4);  return 1;
			case IDC_SQUARE        :  M_CHECK_RADIO (IDC_SQUARE       );  g_mi->f_Waveform->Set_Waveform ( 5);  return 1;
			case IDC_RAMP          :  M_CHECK_RADIO (IDC_RAMP         );  g_mi->f_Waveform->Set_Waveform ( 6);  return 1;
			case IDC_ALT_RAMP      :  M_CHECK_RADIO (IDC_ALT_RAMP     );  g_mi->f_Waveform->Set_Waveform ( 7);  return 1;
			case IDC_BUMPS         :  M_CHECK_RADIO (IDC_BUMPS        );  g_mi->f_Waveform->Set_Waveform ( 8);  return 1;
			case IDC_DBL_BUMPS     :  M_CHECK_RADIO (IDC_DBL_BUMPS    );  g_mi->f_Waveform->Set_Waveform ( 9);  return 1;
			case IDC_ALT_BUMPS     :  M_CHECK_RADIO (IDC_ALT_BUMPS    );  g_mi->f_Waveform->Set_Waveform (10);  return 1;
			case IDC_ALT_BUMPS2    :  M_CHECK_RADIO (IDC_ALT_BUMPS2   );  g_mi->f_Waveform->Set_Waveform (11);  return 1;
			case IDC_SQUARE_BUMPS  :  M_CHECK_RADIO (IDC_SQUARE_BUMPS );  g_mi->f_Waveform->Set_Waveform (12);  return 1;
			case IDC_DBL_BUMPS2    :  M_CHECK_RADIO (IDC_DBL_BUMPS2   );  g_mi->f_Waveform->Set_Waveform (13);  return 1;
			case IDC_DBL_TRIANGLE2 :  M_CHECK_RADIO (IDC_DBL_TRIANGLE2);  g_mi->f_Waveform->Set_Waveform (14);  return 1;

			case IDC_EDIT_DEPTH :
			{
				char  l_Buf [80];

				GetDlgItemText (hDlg, IDC_EDIT_DEPTH, l_Buf, 32);

				int l_Depth = atoi (l_Buf);

				if ((l_Depth >= 0) && (l_Depth <= 10))
				{
					g_mi->f_Fractal->Set_Depth (l_Depth);
					SendDlgItemMessage (hDlg, IDC_SLIDER_DEPTH, TBM_SETPOS, TRUE, l_Depth);
				}

				return 1;
			}
			case IDC_EDIT_EFFECT_HI :
			{
				char  l_Buf [80];

				GetDlgItemText (hDlg, IDC_EDIT_EFFECT_HI, l_Buf, 32);

				float l_Effect_Hi = atof (l_Buf);

				if ((l_Effect_Hi >= 0.0) && (l_Effect_Hi <= 9.0))
				{
					g_mi->f_Fractal->Set_Effect_Hi (l_Effect_Hi);
					SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_HI, TBM_SETPOS, TRUE, (int) (l_Effect_Hi * 1000.0));
				}

				return 1;
			}
			case IDC_EDIT_EFFECT_LO :
			{
				char  l_Buf [80];

				GetDlgItemText (hDlg, IDC_EDIT_EFFECT_LO, l_Buf, 32);

				float l_Effect_Lo = atof (l_Buf);

				if ((l_Effect_Lo >= 0.0) && (l_Effect_Lo <= 9.0))
				{
					g_mi->f_Fractal->Set_Effect_Lo (l_Effect_Lo);
					SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_LO, TBM_SETPOS, TRUE, (int) (l_Effect_Lo * 1000.0));
				}

				return 1;
			}
			case IDC_CHECK_ENABLE_TWIN :
			{
				g_mi->f_Enable_Twin = (SendDlgItemMessage (hDlg, IDC_CHECK_ENABLE_TWIN, BM_GETCHECK, 0, 0) == BST_CHECKED);

				return 1;
			}
			case IDC_CHECK_FRACTAL_BEFORE :
			{
				g_mi->f_Fractal_Before = (SendDlgItemMessage (hDlg, IDC_CHECK_FRACTAL_BEFORE, BM_GETCHECK, 0, 0) == BST_CHECKED);

				return 1;
			}
			case IDC_EDIT_DETUNE :
			{
				char  l_Buf [80];

				GetDlgItemText (hDlg, IDC_EDIT_DETUNE, l_Buf, 32);

				float l_Detune = atof (l_Buf);

				if ((l_Detune >= -12.0) && (l_Detune <= 12.0))
				{
					g_mi->f_Detune_Semitones = l_Detune;
					SendDlgItemMessage (hDlg, IDC_SLIDER_DETUNE, TBM_SETPOS, TRUE, (int) (l_Detune * 10000.0 / 12.0));
				}

				return 1;
			}
			case IDC_EDIT_TWIN_AMP :
			{
				char  l_Buf [80];

				GetDlgItemText (hDlg, IDC_EDIT_TWIN_AMP, l_Buf, 32);

				float l_Twin_Amp = atof (l_Buf);

				if ((l_Twin_Amp >= 0.0) && (l_Twin_Amp <= 1.0))
				{
					g_mi->f_Twin_Amp = l_Twin_Amp;
					SendDlgItemMessage (hDlg, IDC_SLIDER_TWIN_AMP, TBM_SETPOS, TRUE, (int) (l_Twin_Amp * 10000.0));
				}

				return 1;
			}
		}

		break;

	case WM_HSCROLL :
		if ((HWND) lParam == GetDlgItem (hDlg, IDC_SLIDER_DEPTH))
		{
			int   l_Depth     = SendDlgItemMessage (hDlg, IDC_SLIDER_DEPTH, TBM_GETPOS, 0, 0);

			g_mi->f_Fractal->Set_Depth (l_Depth);

			char  l_Buf [80];

			sprintf (l_Buf, "%d", l_Depth    );  SetDlgItemText (hDlg, IDC_EDIT_DEPTH,     l_Buf);
		}
		else if ((HWND) lParam == GetDlgItem (hDlg, IDC_SLIDER_EFFECT_HI))
		{
			float l_Effect_Hi = ((float) SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_HI, TBM_GETPOS, 0, 0)) / 1000.0;

			g_mi->f_Fractal->Set_Effect_Hi (l_Effect_Hi);

			char  l_Buf [80];

			sprintf (l_Buf, "%f", l_Effect_Hi);  SetDlgItemText (hDlg, IDC_EDIT_EFFECT_HI, l_Buf);
		}
		else if ((HWND) lParam == GetDlgItem (hDlg, IDC_SLIDER_EFFECT_LO))
		{
			float l_Effect_Lo = ((float) SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_LO, TBM_GETPOS, 0, 0)) / 1000.0;

			g_mi->f_Fractal->Set_Effect_Lo (((float) SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_LO, TBM_GETPOS, 0, 0)) / 1000.0);

			char  l_Buf [80];

			sprintf (l_Buf, "%f", l_Effect_Lo);  SetDlgItemText (hDlg, IDC_EDIT_EFFECT_LO, l_Buf);
		}
		else if ((HWND) lParam == GetDlgItem (hDlg, IDC_SLIDER_DETUNE))
		{
			g_mi->f_Detune_Semitones = ((float) SendDlgItemMessage (hDlg, IDC_SLIDER_DETUNE, TBM_GETPOS, 0, 0)) * 12.0 / 10000.0;

			char  l_Buf [80];

			sprintf (l_Buf, "%f", g_mi->f_Detune_Semitones);  SetDlgItemText (hDlg, IDC_EDIT_DETUNE, l_Buf);
		}
		else if ((HWND) lParam == GetDlgItem (hDlg, IDC_SLIDER_TWIN_AMP))
		{
			g_mi->f_Twin_Amp = ((float) SendDlgItemMessage (hDlg, IDC_SLIDER_TWIN_AMP, TBM_GETPOS, 0, 0)) / 10000.0;

			char  l_Buf [80];

			sprintf (l_Buf, "%f", g_mi->f_Twin_Amp);  SetDlgItemText (hDlg, IDC_EDIT_TWIN_AMP, l_Buf);
		}

		break;

	}

	return 0;
}

BOOL APIENTRY ModWaveDialog (HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
	case WM_INITDIALOG :
		g_Propsheet_HWnd [3] = hDlg;

		Update_Mod_Waveform_Dlg (g_mi, hDlg);

		return 1;

	case WM_COMMAND :
		switch (LOWORD (wParam))
		{
			//  The CheckRadioButton calls here shouldn't be necessary, but maybe I've hit some
			//  obscure limit.
			//
			//  It might be better to use a combobox or listbox - it will certainly be easier to
			//  extend in the future, and could save space on screen when multiple oscilators are
			//  added (modulators etc).
	
	    case IDC_SINE          :  M_CHECK_RADIO (IDC_SINE         );  g_mi->f_Mod_Waveform->Set_Waveform ( 1);  return 1;
			case IDC_TRIANGLE      :  M_CHECK_RADIO (IDC_TRIANGLE     );  g_mi->f_Mod_Waveform->Set_Waveform ( 2);  return 1;
			case IDC_DBL_TRIANGLE  :  M_CHECK_RADIO (IDC_DBL_TRIANGLE );  g_mi->f_Mod_Waveform->Set_Waveform ( 3);  return 1;
			case IDC_HEX           :  M_CHECK_RADIO (IDC_HEX          );  g_mi->f_Mod_Waveform->Set_Waveform ( 4);  return 1;
			case IDC_SQUARE        :  M_CHECK_RADIO (IDC_SQUARE       );  g_mi->f_Mod_Waveform->Set_Waveform ( 5);  return 1;
			case IDC_RAMP          :  M_CHECK_RADIO (IDC_RAMP         );  g_mi->f_Mod_Waveform->Set_Waveform ( 6);  return 1;
			case IDC_ALT_RAMP      :  M_CHECK_RADIO (IDC_ALT_RAMP     );  g_mi->f_Mod_Waveform->Set_Waveform ( 7);  return 1;
			case IDC_BUMPS         :  M_CHECK_RADIO (IDC_BUMPS        );  g_mi->f_Mod_Waveform->Set_Waveform ( 8);  return 1;
			case IDC_DBL_BUMPS     :  M_CHECK_RADIO (IDC_DBL_BUMPS    );  g_mi->f_Mod_Waveform->Set_Waveform ( 9);  return 1;
			case IDC_ALT_BUMPS     :  M_CHECK_RADIO (IDC_ALT_BUMPS    );  g_mi->f_Mod_Waveform->Set_Waveform (10);  return 1;
			case IDC_ALT_BUMPS2    :  M_CHECK_RADIO (IDC_ALT_BUMPS2   );  g_mi->f_Mod_Waveform->Set_Waveform (11);  return 1;
			case IDC_SQUARE_BUMPS  :  M_CHECK_RADIO (IDC_SQUARE_BUMPS );  g_mi->f_Mod_Waveform->Set_Waveform (12);  return 1;
			case IDC_DBL_BUMPS2    :  M_CHECK_RADIO (IDC_DBL_BUMPS2   );  g_mi->f_Mod_Waveform->Set_Waveform (13);  return 1;
			case IDC_DBL_TRIANGLE2 :  M_CHECK_RADIO (IDC_DBL_TRIANGLE2);  g_mi->f_Mod_Waveform->Set_Waveform (14);  return 1;

      case IDC_RADIO_RAISE_OCTAVES :
        g_mi->f_Mod_Frq_Type = 0;
        CheckRadioButton (hDlg, IDC_RADIO_RAISE_OCTAVES, IDC_RADIO_DIVIDE_FRQ, IDC_RADIO_RAISE_OCTAVES  );
        break;
      case IDC_RADIO_LOWER_OCTAVES :
        g_mi->f_Mod_Frq_Type = 1;
        CheckRadioButton (hDlg, IDC_RADIO_RAISE_OCTAVES, IDC_RADIO_DIVIDE_FRQ, IDC_RADIO_LOWER_OCTAVES  );
        break;
      case IDC_RADIO_RAISE_SEMITONES :
        g_mi->f_Mod_Frq_Type = 2;
        CheckRadioButton (hDlg, IDC_RADIO_RAISE_OCTAVES, IDC_RADIO_DIVIDE_FRQ, IDC_RADIO_RAISE_SEMITONES  );
        break;
      case IDC_RADIO_LOWER_SEMITONES :
        g_mi->f_Mod_Frq_Type = 3;
        CheckRadioButton (hDlg, IDC_RADIO_RAISE_OCTAVES, IDC_RADIO_DIVIDE_FRQ, IDC_RADIO_LOWER_SEMITONES  );
        break;
      case IDC_RADIO_MULTIPLY_FRQ :
        g_mi->f_Mod_Frq_Type = 4;
        CheckRadioButton (hDlg, IDC_RADIO_RAISE_OCTAVES, IDC_RADIO_DIVIDE_FRQ, IDC_RADIO_MULTIPLY_FRQ  );
        break;
      case IDC_RADIO_DIVIDE_FRQ :
        g_mi->f_Mod_Frq_Type = 5;
        CheckRadioButton (hDlg, IDC_RADIO_RAISE_OCTAVES, IDC_RADIO_DIVIDE_FRQ, IDC_RADIO_DIVIDE_FRQ  );
        break;

			case IDC_EDIT_DEPTH :
			{
				char  l_Buf [80];

				GetDlgItemText (hDlg, IDC_EDIT_DEPTH, l_Buf, 32);

				int l_Depth = atoi (l_Buf);

				if ((l_Depth >= 0) && (l_Depth <= 10))
				{
					g_mi->f_Mod_Fractal->Set_Depth (l_Depth);
					SendDlgItemMessage (hDlg, IDC_SLIDER_DEPTH, TBM_SETPOS, TRUE, l_Depth);
				}

				return 1;
			}
			case IDC_EDIT_EFFECT_HI :
			{
				char  l_Buf [80];

				GetDlgItemText (hDlg, IDC_EDIT_EFFECT_HI, l_Buf, 32);

				float l_Effect_Hi = atof (l_Buf);

				if ((l_Effect_Hi >= 0.0) && (l_Effect_Hi <= 9.0))
				{
					g_mi->f_Mod_Fractal->Set_Effect_Hi (l_Effect_Hi);
					SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_HI, TBM_SETPOS, TRUE, (int) (l_Effect_Hi * 1000.0));
				}

				return 1;
			}
			case IDC_EDIT_EFFECT_LO :
			{
				char  l_Buf [80];

				GetDlgItemText (hDlg, IDC_EDIT_EFFECT_LO, l_Buf, 32);

				float l_Effect_Lo = atof (l_Buf);

				if ((l_Effect_Lo >= 0.0) && (l_Effect_Lo <= 9.0))
				{
					g_mi->f_Mod_Fractal->Set_Effect_Lo (l_Effect_Lo);
					SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_LO, TBM_SETPOS, TRUE, (int) (l_Effect_Lo * 1000.0));
				}

				return 1;
			}
			case IDC_EDIT_MODFRQ :
			{
				char  l_Buf [80];

				GetDlgItemText (hDlg, IDC_EDIT_MODFRQ, l_Buf, 32);

				float l_Mod_Frq = atof (l_Buf);

				if ((l_Mod_Frq >= 0.0) && (l_Mod_Frq <= 24.0))
				{
					g_mi->f_Mod_Frq = l_Mod_Frq;
					SendDlgItemMessage (hDlg, IDC_SLIDER_MODFRQ, TBM_SETPOS, TRUE, (int) (l_Mod_Frq * 1000.0));
				}

				return 1;
			}
		}

		break;

	case WM_HSCROLL :
		if ((HWND) lParam == GetDlgItem (hDlg, IDC_SLIDER_DEPTH))
		{
			int   l_Depth     = SendDlgItemMessage (hDlg, IDC_SLIDER_DEPTH, TBM_GETPOS, 0, 0);

			g_mi->f_Mod_Fractal->Set_Depth (l_Depth);

			char  l_Buf [80];

			sprintf (l_Buf, "%d", l_Depth    );  SetDlgItemText (hDlg, IDC_EDIT_DEPTH,     l_Buf);
		}
		else if ((HWND) lParam == GetDlgItem (hDlg, IDC_SLIDER_EFFECT_HI))
		{
			float l_Effect_Hi = ((float) SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_HI, TBM_GETPOS, 0, 0)) / 1000.0;

			g_mi->f_Mod_Fractal->Set_Effect_Hi (l_Effect_Hi);

			char  l_Buf [80];

			sprintf (l_Buf, "%f", l_Effect_Hi);  SetDlgItemText (hDlg, IDC_EDIT_EFFECT_HI, l_Buf);
		}
		else if ((HWND) lParam == GetDlgItem (hDlg, IDC_SLIDER_EFFECT_LO))
		{
			float l_Effect_Lo = ((float) SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_LO, TBM_GETPOS, 0, 0)) / 1000.0;

			g_mi->f_Mod_Fractal->Set_Effect_Lo (((float) SendDlgItemMessage (hDlg, IDC_SLIDER_EFFECT_LO, TBM_GETPOS, 0, 0)) / 1000.0);

			char  l_Buf [80];

			sprintf (l_Buf, "%f", l_Effect_Lo);  SetDlgItemText (hDlg, IDC_EDIT_EFFECT_LO, l_Buf);
		}
		else if ((HWND) lParam == GetDlgItem (hDlg, IDC_SLIDER_MODFRQ))
		{
			g_mi->f_Mod_Frq = ((float) SendDlgItemMessage (hDlg, IDC_SLIDER_MODFRQ, TBM_GETPOS, 0, 0)) / 1000.0;

			char  l_Buf [80];

			sprintf (l_Buf, "%f", g_mi->f_Mod_Frq);  SetDlgItemText (hDlg, IDC_EDIT_MODFRQ, l_Buf);
		}

		break;

	}

	return 0;
}

BOOL APIENTRY AuxBusDialog (HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
	case WM_INITDIALOG :
		g_Propsheet_HWnd [1] = hDlg;

		Update_Aux_Bus_Dlg (g_mi, hDlg);

		return 1;

	case WM_COMMAND :
		switch (LOWORD (wParam))
		{
      case IDC_CHECK_WAV_OPT :
      {
				bool l_Enabled = (SendDlgItemMessage (hDlg, IDC_CHECK_WAV_OPT, BM_GETCHECK, 0, 0) == BST_CHECKED);

        g_mi->f_Waveform->Set_Optimise_Enable (l_Enabled);

        return 1;
      }
      case IDC_EDIT_WAV_OPT :
      {
				char  l_Buf [80];

				GetDlgItemText (hDlg, IDC_EDIT_WAV_OPT, l_Buf, 32);

				int l_Quality = atoi (l_Buf);

        if ((l_Quality > 1) && (l_Quality <= NR_LIB_WAVEFORM_MAX_QUALITY))
        {
          g_mi->f_Waveform->Set_Optimise_Quality (l_Quality);
        }

        return 1;
      }
      case IDC_CHECK_FRACT_OPT :
      {
				bool l_Enabled = (SendDlgItemMessage (hDlg, IDC_CHECK_FRACT_OPT, BM_GETCHECK, 0, 0) == BST_CHECKED);

        g_mi->f_Fractal->Set_Optimise_Enable (l_Enabled);

        return 1;
      }
      case IDC_EDIT_FRACT_OPT :
      {
				char  l_Buf [80];

				GetDlgItemText (hDlg, IDC_EDIT_FRACT_OPT, l_Buf, 32);

				int l_Quality = atoi (l_Buf);

        if ((l_Quality > 1) && (l_Quality <= NR_LIB_FRACTAL_MAX_QUALITY))
        {
          g_mi->f_Fractal->Set_Optimise_Quality (l_Quality);
        }

        return 1;
      }
			case IDC_CHECK_RINGMOD_MAIN :
			{
				g_mi->f_Ringmod_Main = (SendDlgItemMessage (hDlg, IDC_CHECK_RINGMOD_MAIN, BM_GETCHECK, 0, 0) == BST_CHECKED);

				return 1;
			}
			case IDC_CHECK_RINGMOD_TWIN :
			{
				g_mi->f_Ringmod_Twin = (SendDlgItemMessage (hDlg, IDC_CHECK_RINGMOD_TWIN, BM_GETCHECK, 0, 0) == BST_CHECKED);

				return 1;
			}
			case IDC_CHECK_FM_ENABLE :
			{
				g_mi->f_FM_Enable = (SendDlgItemMessage (hDlg, IDC_CHECK_FM_ENABLE, BM_GETCHECK, 0, 0) == BST_CHECKED);

				return 1;
			}
			case IDC_CHECK_AM_ENABLE :
			{
				g_mi->f_AM_Enable = (SendDlgItemMessage (hDlg, IDC_CHECK_AM_ENABLE, BM_GETCHECK, 0, 0) == BST_CHECKED);

				return 1;
			}
			case IDC_CHECK_PWM_ENABLE :
			{
				g_mi->f_PWM_Enable = (SendDlgItemMessage (hDlg, IDC_CHECK_PWM_ENABLE, BM_GETCHECK, 0, 0) == BST_CHECKED);

				return 1;
			}
			case IDC_CHECK_FM_INTERNAL :
			{
				g_mi->f_FM_Internal = (SendDlgItemMessage (hDlg, IDC_CHECK_FM_INTERNAL, BM_GETCHECK, 0, 0) == BST_CHECKED);

				return 1;
			}
			case IDC_CHECK_AM_INTERNAL :
			{
				g_mi->f_AM_Internal = (SendDlgItemMessage (hDlg, IDC_CHECK_AM_INTERNAL, BM_GETCHECK, 0, 0) == BST_CHECKED);

				return 1;
			}
			case IDC_CHECK_PWM_INTERNAL :
			{
				g_mi->f_PWM_Internal = (SendDlgItemMessage (hDlg, IDC_CHECK_PWM_INTERNAL, BM_GETCHECK, 0, 0) == BST_CHECKED);

				return 1;
			}
			case IDC_SET_CHANNEL :
			{
				AB_ShowEditor(NULL, &g_mi->f_Channel, MacInfo.ShortName, cb, g_mi);

				return 1;
			}
			case IDC_DISCONNECT :
			{
				g_mi->f_Channel = -1;

				AB_Disconnect (g_mi);

				return 1;
			}
			case IDC_EDIT_FM_LEVEL :
			{
				char  l_Buf [80];

				GetDlgItemText (hDlg, IDC_EDIT_FM_LEVEL, l_Buf, 32);

				float l_FM_Level = atof (l_Buf);

				if ((l_FM_Level >= 0.0) && (l_FM_Level <= 10.0))
				{
					g_mi->f_FM_Level = l_FM_Level;
					SendDlgItemMessage (hDlg, IDC_SLIDER_FM_LEVEL, TBM_SETPOS, TRUE, (int) M_TO_SLIDER (l_FM_Level * 1000.0));
				}

				return 1;
			}
			case IDC_EDIT_AM_LEVEL :
			{
				char  l_Buf [80];

				GetDlgItemText (hDlg, IDC_EDIT_AM_LEVEL, l_Buf, 32);

				float l_AM_Level = atof (l_Buf);

				if ((l_AM_Level >= 0.0) && (l_AM_Level <= 1.0))
				{
					g_mi->f_AM_Level = l_AM_Level;
					SendDlgItemMessage (hDlg, IDC_SLIDER_AM_LEVEL, TBM_SETPOS, TRUE, (int) M_TO_SLIDER (l_AM_Level * 10000.0));
				}

				return 1;
			}
			case IDC_EDIT_PWM_LEVEL :
			{
				char  l_Buf [80];

				GetDlgItemText (hDlg, IDC_EDIT_PWM_LEVEL, l_Buf, 32);

				float l_PWM_Level = atof (l_Buf);

				if ((l_PWM_Level >= 0.0) && (l_PWM_Level <= 1.0))
				{
					g_mi->f_PWM_Level = l_PWM_Level;
					SendDlgItemMessage (hDlg, IDC_SLIDER_PWM_LEVEL, TBM_SETPOS, TRUE, (int) M_TO_SLIDER (l_PWM_Level * 10000.0));
				}

				return 1;
			}
		}

		break;

	case WM_HSCROLL :
		if ((HWND) lParam == GetDlgItem (hDlg, IDC_SLIDER_FM_LEVEL))
		{
			g_mi->f_FM_Level = M_FROM_SLIDER (SendDlgItemMessage (hDlg, IDC_SLIDER_FM_LEVEL, TBM_GETPOS, 0, 0)) / 1000.0;

			char  l_Buf [80];

			sprintf (l_Buf, "%f", g_mi->f_FM_Level);  SetDlgItemText (hDlg, IDC_EDIT_FM_LEVEL, l_Buf);
		}
		else if ((HWND) lParam == GetDlgItem (hDlg, IDC_SLIDER_AM_LEVEL))
		{
			g_mi->f_AM_Level = M_FROM_SLIDER (SendDlgItemMessage (hDlg, IDC_SLIDER_AM_LEVEL, TBM_GETPOS, 0, 0)) / 10000.0;

			char  l_Buf [80];

			sprintf (l_Buf, "%f", g_mi->f_AM_Level);  SetDlgItemText (hDlg, IDC_EDIT_AM_LEVEL, l_Buf);
		}
		else if ((HWND) lParam == GetDlgItem (hDlg, IDC_SLIDER_PWM_LEVEL))
		{
			g_mi->f_PWM_Level = M_FROM_SLIDER (SendDlgItemMessage (hDlg, IDC_SLIDER_PWM_LEVEL, TBM_GETPOS, 0, 0)) / 10000.0;

			char  l_Buf [80];

			sprintf (l_Buf, "%f", g_mi->f_PWM_Level);  SetDlgItemText (hDlg, IDC_EDIT_PWM_LEVEL, l_Buf);
		}

		break;
	}

	return 0;
}

LV_COLUMN g_Listview_Cols [] =
	{
		{LVCF_TEXT | LVCF_WIDTH | LVCF_SUBITEM, LVCFMT_LEFT, 100, "ID",            2, 0},
		{LVCF_TEXT | LVCF_WIDTH | LVCF_SUBITEM, LVCFMT_LEFT, 100, "Program Name", 12, 1}
	};

#define NUM_CONTROL_CONFIGS 10

c_Control_Config g_Control_Configs [NUM_CONTROL_CONFIGS] =
{
  {0, "Volume"             },
  {1, "Attack"             },
  {2, "Decay"              },
  {3, "Sustain"            },
  {4, "Release"            },
  {5, "Bend Rate"          },
  {6, "Fractal Effect Low" },
  {7, "Fractal Effect High"},
  {8, "Modulator Fractal Effect Low" },
  {9, "Modulator Fractal Effect High"}
};

BOOL APIENTRY RxBroadcastDialog (HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
	case WM_INITDIALOG :

		CheckDlgButton (hDlg, IDC_CHECK_LISTEN_ENABLE, g_mi->f_Listen_Enabled );

		{
			char  l_Buf [80];

			sprintf (l_Buf, "%d", g_mi->f_Listen_Channel);  SetDlgItemText (hDlg, IDC_EDIT_LISTEN_CHAN,  l_Buf);

			EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_PRG_CREATE), FALSE);
			EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_PRG_DELETE), FALSE);
			EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_PRG_UPDATE), FALSE);
			EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_PRG_SELECT), FALSE);
			EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_ID_CREATE ), FALSE);
			EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_ID_DELETE ), FALSE);

			SendDlgItemMessage (hDlg, IDC_SPIN_LISTEN_CHAN, UDM_SETBUDDY, (WPARAM) (HWND) GetDlgItem (hDlg, IDC_EDIT_LISTEN_CHAN), 0L);
			SendDlgItemMessage (hDlg, IDC_SPIN_LISTEN_CHAN, UDM_SETRANGE, 0, MAKELONG(7, 0));

			SendDlgItemMessage (hDlg, IDC_COMBO_PROG_NAME, CB_LIMITTEXT, 31, 0L);
			SendDlgItemMessage (hDlg, IDC_COMBO_PRG_IDS,   CB_LIMITTEXT, 31, 0L);

			ListView_InsertColumn (GetDlgItem (hDlg, IDC_LIST_PROG_IDS), 0, &g_Listview_Cols [0]);
			ListView_InsertColumn (GetDlgItem (hDlg, IDC_LIST_PROG_IDS), 1, &g_Listview_Cols [1]);
 
			c_Program::Fill_Program_Combo (g_mi, hDlg);
			c_Program::Fill_ID_List       (g_mi, hDlg);
		}

		return 1;
	case WM_NOTIFY :
	{
		NMHDR *l_Notify = (NMHDR *) lParam;

		if (   (l_Notify->hwndFrom == GetDlgItem (hDlg, IDC_LIST_PROG_IDS))
			&& (l_Notify->code     == LVN_GETDISPINFO                     ))
		{
			LV_DISPINFO *l_Disp_Info = (LV_DISPINFO *) lParam;

			if (l_Disp_Info->item.iSubItem == 1)
			{
        if (g_mi->f_Program_IDs [l_Disp_Info->item.lParam] == -1)
        {
          l_Disp_Info->item.pszText = "";
        }
        else
        {
				  l_Disp_Info->item.pszText = g_mi->f_Programs [g_mi->f_Program_IDs [l_Disp_Info->item.lParam]].f_Name;
        }
			}
		}

		break;
	}
	case WM_COMMAND :
		switch (LOWORD (wParam))
		{
      case IDC_CONFIG_CTRLS :
      {
        NR_Dialog_Enable_Channel (g_mi, &g_mi->f_Listen_Enabled, &g_mi->f_Listen_Channel,
                                  NUM_CONTROL_CONFIGS, g_Control_Configs                 );


        return 1;
      }
			case IDC_CHECK_LISTEN_ENABLE :
			{
				bool l_Enabled = (SendDlgItemMessage (hDlg, IDC_CHECK_LISTEN_ENABLE, BM_GETCHECK, 0, 0) == BST_CHECKED);

				if (g_mi->f_Listen_Enabled)
				{
					if (!l_Enabled)
					{
						NR_Stop_Listening  (g_mi->f_Listen_Channel, g_mi);
					}
				}
				else
				{
					if (l_Enabled)
					{
						NR_Start_Listening (g_mi->f_Listen_Channel, g_mi);
					}
				}

				g_mi->f_Listen_Enabled = l_Enabled;
				return 1;
			}
			case IDC_EDIT_LISTEN_CHAN :
			{
				char  l_Buf [80];

				GetDlgItemText (hDlg, IDC_EDIT_LISTEN_CHAN, l_Buf, 32);

				int l_Channel = atoi (l_Buf);

				if ((l_Channel >= 0) && (l_Channel <= 7) && (l_Channel != g_mi->f_Listen_Channel))
				{
					if (g_mi->f_Listen_Enabled)
					{
						NR_Stop_Listening  (g_mi->f_Listen_Channel, g_mi);
						g_mi->f_Listen_Channel = l_Channel;
						NR_Start_Listening (l_Channel, g_mi);
					}
					else
					{
						g_mi->f_Listen_Channel = l_Channel;
					}
				}

				return 1;
			}
			case IDC_COMBO_PROG_NAME :
			{
				char  l_Buf [MAX_PROGRAM_NAME];
				int   l_Prog, l_ID;

				if (HIWORD(wParam) == CBN_EDITUPDATE)
				{
					GetDlgItemText (hDlg, IDC_COMBO_PROG_NAME, l_Buf, MAX_PROGRAM_NAME);

					l_Prog = c_Program::Find (g_mi, l_Buf);
				}
				else if (HIWORD(wParam) == CBN_SELENDOK)
				{
			        WORD n = (WORD) SendDlgItemMessage (hDlg, IDC_COMBO_PROG_NAME, CB_GETCURSEL, 0, 0L);
			        SendDlgItemMessage (hDlg, IDC_COMBO_PROG_NAME, CB_GETLBTEXT, n, (LONG) (LPSTR) l_Buf);

					l_Prog = c_Program::Find (g_mi, l_Buf);
				}
				else
				{
					return 1;
				}

				c_Program::Fill_ID_Combo (g_mi, hDlg, l_Prog);

				EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_PRG_CREATE),
					          (l_Prog == -1) && (l_Buf[0] != 0) && (g_mi->f_Num_Programs < MAX_PROGRAMS));

				EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_PRG_DELETE),
					          (l_Prog != -1)                           );

				EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_PRG_UPDATE),
					          (l_Prog != -1)                           );

				EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_PRG_SELECT),
					          (l_Prog != -1)                           );

				GetDlgItemText (hDlg, IDC_COMBO_PRG_IDS, l_Buf, MAX_PROGRAM_NAME);

				l_ID = atoi (l_Buf);

				EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_ID_CREATE ),
					          (l_Prog != -1) && (l_Buf [0] != 0) && (g_mi->f_Program_IDs [l_ID] == -1    ) );

				EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_ID_DELETE ),
					          (l_Prog != -1) && (l_Buf [0] != 0) && (g_mi->f_Program_IDs [l_ID] == l_Prog) );

				return 1;
			}
			case IDC_COMBO_PRG_IDS :
			{
				char  l_Buf [MAX_PROGRAM_NAME];
				int   l_Prog, l_ID;

				GetDlgItemText (hDlg, IDC_COMBO_PROG_NAME, l_Buf, MAX_PROGRAM_NAME);

				l_Prog = c_Program::Find (g_mi, l_Buf);

				if (HIWORD(wParam) == CBN_EDITUPDATE)
				{
					GetDlgItemText (hDlg, IDC_COMBO_PRG_IDS, l_Buf, MAX_PROGRAM_NAME);
				}
				else if (HIWORD(wParam) == CBN_SELENDOK)
				{
			        WORD n = (WORD) SendDlgItemMessage (hDlg, IDC_COMBO_PRG_IDS, CB_GETCURSEL, 0, 0L);
			        SendDlgItemMessage (hDlg, IDC_COMBO_PRG_IDS, CB_GETLBTEXT, n, (LONG) (LPSTR) l_Buf);
				}
				else
				{
					return 1;
				}

				l_ID = atoi (l_Buf);

				EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_ID_CREATE ),
					          (l_Prog != -1) && (l_Buf [0] != 0) && (g_mi->f_Program_IDs [l_ID] == -1    ) );

				EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_ID_DELETE ),
					          (l_Prog != -1) && (l_Buf [0] != 0) && (g_mi->f_Program_IDs [l_ID] == l_Prog) );

				return 1;
			}
			case IDC_BUTTON_PRG_CREATE :
			{
				char  l_Buf [MAX_PROGRAM_NAME];

				GetDlgItemText (hDlg, IDC_COMBO_PROG_NAME, l_Buf, MAX_PROGRAM_NAME);

				int l_Prog = c_Program::Find (g_mi, l_Buf);

				if ((l_Prog == -1) && (l_Buf[0] != 0) && (g_mi->f_Num_Programs < MAX_PROGRAMS))
				{
					strcpy (g_mi->f_Programs [g_mi->f_Num_Programs].f_Name, l_Buf);
					g_mi->f_Programs [g_mi->f_Num_Programs].Get_From_Current (g_mi);
					g_mi->f_Num_Programs++;

					EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_PRG_CREATE), FALSE);
					EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_PRG_DELETE), TRUE );
					EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_PRG_UPDATE), TRUE );
					EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_PRG_SELECT), TRUE );
					EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_ID_CREATE ), FALSE);
					EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_ID_DELETE ), FALSE);

					c_Program::Fill_Program_Combo (g_mi, hDlg);
					c_Program::Fill_ID_Combo      (g_mi, hDlg, g_mi->f_Num_Programs - 1);
					c_Program::Fill_ID_List       (g_mi, hDlg);

					SetDlgItemText (hDlg, IDC_COMBO_PROG_NAME, l_Buf);
				}

				return 1;
			}
			case IDC_BUTTON_PRG_DELETE :
			{
				char  l_Buf [MAX_PROGRAM_NAME];

				GetDlgItemText (hDlg, IDC_COMBO_PROG_NAME, l_Buf, MAX_PROGRAM_NAME);

				int l_Prog = c_Program::Find (g_mi, l_Buf);

				if (l_Prog != -1)
				{
					g_mi->f_Num_Programs--;

					if (l_Prog != g_mi->f_Num_Programs)
					{
						g_mi->f_Programs[l_Prog] = g_mi->f_Programs[l_Prog];  //  Note shallow copy
					}

					for (int i = 0; i < MAX_PROGRAM_IDS; i++)
					{
						if (g_mi->f_Program_IDs [i] == l_Prog)
						{
							g_mi->f_Program_IDs [i] = -1;
						}
					}

					EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_PRG_CREATE), TRUE );
					EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_PRG_DELETE), FALSE);
					EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_PRG_UPDATE), FALSE);
					EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_PRG_SELECT), FALSE);
					EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_ID_CREATE ), FALSE);
					EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_ID_DELETE ), FALSE);

					c_Program::Fill_Program_Combo (g_mi, hDlg);
					c_Program::Fill_ID_Combo (g_mi, hDlg, l_Prog);
					c_Program::Fill_ID_List       (g_mi, hDlg);

					SetDlgItemText (hDlg, IDC_COMBO_PROG_NAME, l_Buf);
				}

				return 1;
			}
			case IDC_BUTTON_PRG_UPDATE :
			{
				char  l_Buf [MAX_PROGRAM_NAME];

				GetDlgItemText (hDlg, IDC_COMBO_PROG_NAME, l_Buf, MAX_PROGRAM_NAME);

				int l_Prog = c_Program::Find (g_mi, l_Buf);

				if (l_Prog != -1)
				{
					g_mi->f_Programs [l_Prog].Get_From_Current (g_mi);
					c_Program::Fill_ID_Combo (g_mi, hDlg, l_Prog);
					c_Program::Fill_ID_List       (g_mi, hDlg);
				}

				return 1;
			}
			case IDC_BUTTON_PRG_SELECT :
			{
				char  l_Buf [MAX_PROGRAM_NAME];

				GetDlgItemText (hDlg, IDC_COMBO_PROG_NAME, l_Buf, MAX_PROGRAM_NAME);

				int l_Prog = c_Program::Find (g_mi, l_Buf);

				if (l_Prog != -1)
				{
					g_mi->f_Programs [l_Prog].Set_As_Current (g_mi);
					c_Program::Fill_ID_Combo (g_mi, hDlg, l_Prog);
					c_Program::Fill_ID_List       (g_mi, hDlg);
					Update_All_Dialogs ();
				}

				return 1;
			}
			case IDC_BUTTON_ID_CREATE :
			{
				char  l_Buf [MAX_PROGRAM_NAME];

				GetDlgItemText (hDlg, IDC_COMBO_PROG_NAME, l_Buf, MAX_PROGRAM_NAME);

				int l_Prog = c_Program::Find (g_mi, l_Buf);

				GetDlgItemText (hDlg, IDC_COMBO_PRG_IDS, l_Buf, MAX_PROGRAM_NAME);

				int l_ID = atoi (l_Buf);

				if ((l_Prog != -1) && (l_Buf [0] != 0) && (g_mi->f_Program_IDs [l_ID] == -1))
				{
					g_mi->f_Program_IDs [l_ID] = l_Prog;

					EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_ID_CREATE ),
						          (l_Prog != -1) && (l_Buf [0] != 0) && (g_mi->f_Program_IDs [l_ID] == -1    ) );

					EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_ID_DELETE ),
						          (l_Prog != -1) && (l_Buf [0] != 0) && (g_mi->f_Program_IDs [l_ID] == l_Prog) );

					c_Program::Fill_ID_Combo (g_mi, hDlg, l_Prog);
					c_Program::Fill_ID_List       (g_mi, hDlg);
				}

				return 1;
			}
			case IDC_BUTTON_ID_DELETE :
			{
				char  l_Buf [MAX_PROGRAM_NAME];

				GetDlgItemText (hDlg, IDC_COMBO_PROG_NAME, l_Buf, MAX_PROGRAM_NAME);

				int l_Prog = c_Program::Find (g_mi, l_Buf);

				GetDlgItemText (hDlg, IDC_COMBO_PRG_IDS, l_Buf, MAX_PROGRAM_NAME);

				int l_ID = atoi (l_Buf);

				if ((l_Prog != -1) && (l_Buf [0] != 0) && (g_mi->f_Program_IDs [l_ID] == l_Prog))
				{
					g_mi->f_Program_IDs [l_ID] = -1;

					EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_ID_CREATE ),
						          (l_Prog != -1) && (l_Buf [0] != 0) && (g_mi->f_Program_IDs [l_ID] == -1    ) );

					EnableWindow (GetDlgItem (hDlg, IDC_BUTTON_ID_DELETE ),
						          (l_Prog != -1) && (l_Buf [0] != 0) && (g_mi->f_Program_IDs [l_ID] == l_Prog) );

					c_Program::Fill_ID_Combo (g_mi, hDlg, l_Prog);
					c_Program::Fill_ID_List       (g_mi, hDlg);
				}

				return 1;
			}
		}
	}

	return 0;
}


BOOL APIENTRY EnvelDialog (HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
	case WM_INITDIALOG :
		g_Propsheet_HWnd [2] = hDlg;

		Update_Envel_Dlg (g_mi, hDlg);

		return 1;

	case WM_COMMAND :
		switch (LOWORD (wParam))
		{
      M_LOG_SLIDER_EDIT_NOTIFY(IDC_SLIDER_ATTACK, IDC_EDIT_ATTACK, g_mi->f_Level->Set_Attack  (l_Temp))
      M_LOG_SLIDER_EDIT_NOTIFY(IDC_SLIDER_DECAY,  IDC_EDIT_DECAY,  g_mi->f_Level->Set_Decay   (l_Temp))
      M_LIN_SLIDER_EDIT_NOTIFY(IDC_SLIDER_SUSTAIN,IDC_EDIT_SUSTAIN,g_mi->f_Level->Set_Sustain (l_Temp))
      M_LOG_SLIDER_EDIT_NOTIFY(IDC_SLIDER_RELEASE,IDC_EDIT_RELEASE,g_mi->f_Level->Set_Release (l_Temp))
      M_LOG_SLIDER_EDIT_NOTIFY(IDC_SLIDER_BEND,   IDC_EDIT_BEND,   g_mi->f_Pitch->Set_Attack  (l_Temp))
		}

		break;

	case WM_HSCROLL :
         M_LOG_SLIDER_SLIDE_NOTIFY(IDC_SLIDER_ATTACK, IDC_EDIT_ATTACK, g_mi->f_Level->Set_Attack  (l_Temp))
    else M_LOG_SLIDER_SLIDE_NOTIFY(IDC_SLIDER_DECAY,  IDC_EDIT_DECAY,  g_mi->f_Level->Set_Decay   (l_Temp))
    else M_LIN_SLIDER_SLIDE_NOTIFY(IDC_SLIDER_SUSTAIN,IDC_EDIT_SUSTAIN,g_mi->f_Level->Set_Sustain (l_Temp))
    else M_LOG_SLIDER_SLIDE_NOTIFY(IDC_SLIDER_RELEASE,IDC_EDIT_RELEASE,g_mi->f_Level->Set_Release (l_Temp))
    else M_LOG_SLIDER_SLIDE_NOTIFY(IDC_SLIDER_BEND,   IDC_EDIT_BEND,   g_mi->f_Pitch->Set_Attack  (l_Temp))
	}

	return 0;
}

BOOL APIENTRY ModEnvelDialog (HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam)
{
	switch (uMsg)
	{
	case WM_INITDIALOG :
		g_Propsheet_HWnd [4] = hDlg;

		Update_Mod_Envel_Dlg (g_mi, hDlg);

		return 1;

	case WM_COMMAND :
		switch (LOWORD (wParam))
		{
      M_LOG_SLIDER_EDIT_NOTIFY(IDC_SLIDER_ATTACK, IDC_EDIT_ATTACK, g_mi->f_Mod_Level->Set_Attack  (l_Temp))
      M_LOG_SLIDER_EDIT_NOTIFY(IDC_SLIDER_DECAY,  IDC_EDIT_DECAY,  g_mi->f_Mod_Level->Set_Decay   (l_Temp))
      M_LIN_SLIDER_EDIT_NOTIFY(IDC_SLIDER_SUSTAIN,IDC_EDIT_SUSTAIN,g_mi->f_Mod_Level->Set_Sustain (l_Temp))
      M_LOG_SLIDER_EDIT_NOTIFY(IDC_SLIDER_RELEASE,IDC_EDIT_RELEASE,g_mi->f_Mod_Level->Set_Release (l_Temp))
      M_LIN_SLIDER_EDIT_NOTIFY(IDC_SLIDER_START,  IDC_EDIT_START,  g_mi->f_Mod_Level->Set_Start   (l_Temp))
      M_LIN_SLIDER_EDIT_NOTIFY(IDC_SLIDER_PEAK,   IDC_EDIT_PEAK,   g_mi->f_Mod_Level->Set_Peak    (l_Temp))
      M_LIN_SLIDER_EDIT_NOTIFY(IDC_SLIDER_END,    IDC_EDIT_END,    g_mi->f_Mod_Level->Set_End     (l_Temp))
		}

		break;

	case WM_HSCROLL :
         M_LOG_SLIDER_SLIDE_NOTIFY(IDC_SLIDER_ATTACK, IDC_EDIT_ATTACK, g_mi->f_Mod_Level->Set_Attack  (l_Temp))
    else M_LOG_SLIDER_SLIDE_NOTIFY(IDC_SLIDER_DECAY,  IDC_EDIT_DECAY,  g_mi->f_Mod_Level->Set_Decay   (l_Temp))
    else M_LIN_SLIDER_SLIDE_NOTIFY(IDC_SLIDER_SUSTAIN,IDC_EDIT_SUSTAIN,g_mi->f_Mod_Level->Set_Sustain (l_Temp))
    else M_LOG_SLIDER_SLIDE_NOTIFY(IDC_SLIDER_RELEASE,IDC_EDIT_RELEASE,g_mi->f_Mod_Level->Set_Release (l_Temp))
    else M_LIN_SLIDER_SLIDE_NOTIFY(IDC_SLIDER_START,  IDC_EDIT_START,  g_mi->f_Mod_Level->Set_Start   (l_Temp))
    else M_LIN_SLIDER_SLIDE_NOTIFY(IDC_SLIDER_PEAK,   IDC_EDIT_PEAK,   g_mi->f_Mod_Level->Set_Peak    (l_Temp))
    else M_LIN_SLIDER_SLIDE_NOTIFY(IDC_SLIDER_END,    IDC_EDIT_END,    g_mi->f_Mod_Level->Set_End     (l_Temp))
	}

	return 0;
}

void mi::Command(int const i)
{
	switch (i)
	{
	case 0 :  //  Edit Ninereeds NRS04
		{
			InitCommonControls ();

			PROPSHEETPAGE l_Pages[7];

			l_Pages[0].dwSize      = sizeof(PROPSHEETPAGE);
			l_Pages[0].dwFlags     = PSP_DEFAULT | PSP_USETITLE;
			l_Pages[0].hInstance   = dllInstance;
			l_Pages[0].pszTemplate = MAKEINTRESOURCE(IDD_NRS04);
			l_Pages[0].pszIcon     = NULL;
			l_Pages[0].pszTitle    = "Carrier Waveform";
			l_Pages[0].pfnDlgProc  = (DLGPROC) &MainDialog;
			l_Pages[0].lParam      = 0;
			l_Pages[0].pfnCallback = NULL;
			l_Pages[0].pcRefParent = NULL;

			l_Pages[1].dwSize      = sizeof(PROPSHEETPAGE);
			l_Pages[1].dwFlags     = PSP_DEFAULT;
			l_Pages[1].hInstance   = dllInstance;
			l_Pages[1].pszTemplate = MAKEINTRESOURCE(IDD_ENVEL);
			l_Pages[1].pszIcon     = NULL;
			l_Pages[1].pszTitle    = NULL;
			l_Pages[1].pfnDlgProc  = (DLGPROC) &EnvelDialog;
			l_Pages[1].lParam      = 0;
			l_Pages[1].pfnCallback = NULL;
			l_Pages[1].pcRefParent = NULL;

			l_Pages[2].dwSize      = sizeof(PROPSHEETPAGE);
			l_Pages[2].dwFlags     = PSP_DEFAULT;
			l_Pages[2].hInstance   = dllInstance;
			l_Pages[2].pszTemplate = MAKEINTRESOURCE(IDD_MOD_WAVE);
			l_Pages[2].pszIcon     = NULL;
			l_Pages[2].pszTitle    = NULL;
			l_Pages[2].pfnDlgProc  = (DLGPROC) &ModWaveDialog;
			l_Pages[2].lParam      = 0;
			l_Pages[2].pfnCallback = NULL;
			l_Pages[2].pcRefParent = NULL;

			l_Pages[3].dwSize      = sizeof(PROPSHEETPAGE);
			l_Pages[3].dwFlags     = PSP_DEFAULT;
			l_Pages[3].hInstance   = dllInstance;
			l_Pages[3].pszTemplate = MAKEINTRESOURCE(IDD_MOD_ENVEL);
			l_Pages[3].pszIcon     = NULL;
			l_Pages[3].pszTitle    = NULL;
			l_Pages[3].pfnDlgProc  = (DLGPROC) &ModEnvelDialog;
			l_Pages[3].lParam      = 0;
			l_Pages[3].pfnCallback = NULL;
			l_Pages[3].pcRefParent = NULL;

			l_Pages[4].dwSize      = sizeof(PROPSHEETPAGE);
			l_Pages[4].dwFlags     = PSP_DEFAULT;
			l_Pages[4].hInstance   = dllInstance;
			l_Pages[4].pszTemplate = MAKEINTRESOURCE(IDD_AUXBUS_FX);
			l_Pages[4].pszIcon     = NULL;
			l_Pages[4].pszTitle    = NULL;
			l_Pages[4].pfnDlgProc  = (DLGPROC) &AuxBusDialog;
			l_Pages[4].lParam      = 0;
			l_Pages[4].pfnCallback = NULL;
			l_Pages[4].pcRefParent = NULL;

			l_Pages[5].dwSize      = sizeof(PROPSHEETPAGE);
			l_Pages[5].dwFlags     = PSP_DEFAULT;
			l_Pages[5].hInstance   = dllInstance;
			l_Pages[5].pszTemplate = MAKEINTRESOURCE(IDD_RX_BROADCAST);
			l_Pages[5].pszIcon     = NULL;
			l_Pages[5].pszTitle    = NULL;
			l_Pages[5].pfnDlgProc  = (DLGPROC) &RxBroadcastDialog;
			l_Pages[5].lParam      = 0;
			l_Pages[5].pfnCallback = NULL;
			l_Pages[5].pcRefParent = NULL;

			l_Pages[6].dwSize      = sizeof(PROPSHEETPAGE);
			l_Pages[6].dwFlags     = PSP_DEFAULT;
			l_Pages[6].hInstance   = dllInstance;
			l_Pages[6].pszTemplate = MAKEINTRESOURCE(IDD_ABOUT);
			l_Pages[6].pszIcon     = NULL;
			l_Pages[6].pszTitle    = NULL;
			l_Pages[6].pfnDlgProc  = (DLGPROC) NULL;
			l_Pages[6].lParam      = 0;
			l_Pages[6].pfnCallback = NULL;
			l_Pages[6].pcRefParent = NULL;

			PROPSHEETHEADER l_Prop_Sheet;

			l_Prop_Sheet.dwSize      = sizeof(PROPSHEETHEADER);
			l_Prop_Sheet.dwFlags     = PSH_DEFAULT | PSH_PROPSHEETPAGE;
			l_Prop_Sheet.hwndParent  = GetForegroundWindow ();
			l_Prop_Sheet.hInstance   = dllInstance;
			l_Prop_Sheet.hIcon       = NULL;
			l_Prop_Sheet.pszCaption  = (LPCTSTR) "Ninereeds NRS04, By Steve Horne";
			l_Prop_Sheet.nPages      = 7;
			l_Prop_Sheet.nStartPage  = 0;
			l_Prop_Sheet.ppsp        = (LPCPROPSHEETPAGE) l_Pages;
			l_Prop_Sheet.pfnCallback = NULL;

			g_mi = this;

			//DialogBox (dllInstance, MAKEINTRESOURCE (IDD_NRS04), NULL, (DLGPROC) &MainDialog);

			g_Propsheet_HWnd [0] = NULL;
			g_Propsheet_HWnd [1] = NULL;
			g_Propsheet_HWnd [2] = NULL;
			g_Propsheet_HWnd [3] = NULL;  //  Modulator Waveform
			g_Propsheet_HWnd [4] = NULL;  //  Modulator Envelope

			PropertySheet ((LPCPROPSHEETHEADER) &l_Prop_Sheet);

			//  This is in case of asynchronous events that could affect the property sheet if it
			//  is displayed, but do not care if the property sheet is not displayed.
			//
			//  Not going ahead at present because some assurances are needed that the access will
			//  not occur between the property sheet closing and these variables being cleared.
			//
			//  Is there a notification that warns of the property sheet (or of a single page) closing?

			g_mi = NULL;

			g_Propsheet_HWnd [0] = NULL;
			g_Propsheet_HWnd [1] = NULL;
			g_Propsheet_HWnd [2] = NULL;
			g_Propsheet_HWnd [3] = NULL;
			g_Propsheet_HWnd [4] = NULL;
		}
		break;

	}
}
